"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.request = request;
const axios_1 = __importDefault(require("axios"));
const form_data_1 = __importDefault(require("form-data"));
/**
 * Detects the number of parameters expected by a callback function
 * @param callback The callback function to analyze
 * @returns The number of parameters the callback expects
 */
function detectCallbackParams(callback) {
    const callbackStr = callback.toString();
    const paramMatch = callbackStr.match(/\([^)]*\)/);
    if (!paramMatch)
        return 3; // Default to 3 params for safety
    const params = paramMatch[0].slice(1, -1).split(',').map(p => p.trim()).filter(p => p.length > 0);
    // If callback has exactly 2 parameters, return 2
    if (params.length === 2)
        return 2;
    // Default to 3 parameters for compatibility
    return 3;
}
/**
 * Enhanced request function that handles both 2-param and 3-param callback patterns
 * @param options Request configuration options
 * @param callback Callback function (supports both 2 and 3 parameter patterns)
 */
function request(options, callback) {
    if (options.log) {
        console.log(`[AxiosRequest] ${options.method || 'GET'} ${options.url}`);
    }
    const method = (options.method || 'GET').toLowerCase();
    const config = {
        method: method,
        url: options.url || '',
        headers: options.headers || {},
        params: options.params || options.qs,
        validateStatus: () => true,
        timeout: 30000 // 30 second timeout
    };
    // Handle auth if provided
    if (options.auth) {
        config.auth = {
            username: options.auth.user,
            password: options.auth.pass
        };
    }
    // Handle formData (file uploads)
    if (options.formData) {
        const formData = new form_data_1.default();
        for (const key in options.formData) {
            const value = options.formData[key];
            if (value && typeof value === 'object' && 'value' in value) {
                // Handle file stream
                const appendOptions = {};
                if (value.options) {
                    if (value.options.filename !== undefined) {
                        appendOptions.filename = value.options.filename;
                    }
                    if (value.options.contentType !== null && value.options.contentType !== undefined) {
                        appendOptions.contentType = value.options.contentType;
                    }
                }
                formData.append(key, value.value, appendOptions);
            }
            else {
                // Normal field
                formData.append(key, value);
            }
        }
        config.data = formData;
        config.headers = { ...config.headers, ...formData.getHeaders() };
    }
    // Handle x-www-form-urlencoded
    else if (options.form) {
        config.data = options.form;
        config.headers = { ...config.headers, 'Content-Type': 'application/x-www-form-urlencoded' };
    }
    else if (options.body && ['post', 'put', 'patch', 'delete'].includes(method)) {
        if (typeof options.body === 'string') {
            try {
                config.data = JSON.parse(options.body);
                config.headers = { ...config.headers, 'Content-Type': 'application/json' };
            }
            catch (error) {
                // If JSON parsing fails, send the raw string as data
                config.data = options.body;
                config.headers = { ...config.headers, 'Content-Type': 'text/plain' };
            }
        }
        else {
            config.data = options.body;
            config.headers = { ...config.headers, 'Content-Type': 'application/json' };
        }
    }
    // Detect callback parameter count
    const callbackParamCount = detectCallbackParams(callback);
    // Log config only for VerifyProfile requests (URL contains "/Verifications")
    /* if (config.url && config.url.includes('/Verifications')) {
      console.log("VerifyProfile config:", config);
      return;
  
    } */
    (0, axios_1.default)(config)
        .then((response) => {
        const res = {
            statusCode: response.status,
            headers: response.headers,
            body: typeof response.data === 'string' ? response.data : JSON.stringify(response.data)
        };
        if (options.log) {
            console.log(`[AxiosRequest] SUCCESS: ${method.toUpperCase()} ${options.url} -> ${res.statusCode}`);
        }
        // Call callback with appropriate number of parameters
        if (callbackParamCount === 2) {
            callback(null, res);
        }
        else {
            callback(null, res, res.body);
        }
    })
        .catch((error) => {
        const errorMessage = error.response?.data?.message || error.message || 'Unknown error';
        if (options.log) {
            console.error(`[AxiosRequest] ERROR: ${method.toUpperCase()} ${options.url}`, errorMessage);
        }
        // Call callback with appropriate number of parameters
        if (callbackParamCount === 2) {
            callback(error, undefined);
        }
        else {
            callback(error, undefined, undefined);
        }
    });
}
