import { Injectable } from '@angular/core';

@Injectable({
  providedIn: 'root'
})
export class ZohoSalesIQService {
  private scriptLoaded = false;
  private scriptUrl = 'https://salesiq.zoho.com/widget';
  private initPromise: Promise<void> | null = null;
  config: { widgetCode: string; disabledPaths: string[]; };

  constructor() {
    this.setupAttributionTracking();
  }

  async initialize(config: { widgetCode: string; disabledPaths: string[] }): Promise<void> {
    this.config = config;

    if (this.initPromise) return this.initPromise;

    this.initPromise = new Promise((resolve) => {
      if (this.scriptLoaded || this.config.disabledPaths.includes(window.location.pathname)) {
        resolve();
        return;
      }

      const originalAppendChild = document.head.appendChild.bind(document.head);
      document.head.appendChild = (element: any) => {
        if (element.src?.includes('embedfloat/js/float') || element.src?.includes('embedfloat/css/float')) {
          console.log('Prevented loading of chat module:', element.src);
          return element;
        }
        return originalAppendChild(element);
      };
      
      (window as any).$zoho = (window as any).$zoho || {};
      (window as any).$zoho.salesiq = (window as any).$zoho.salesiq || {
        widgetcode: this.config.widgetCode,
        values: {},
        ready: () => {
          this.handleWidgetReady();
          resolve();
        }
      };

      (window as any)._SIQ_CHAT_WIDGET = false;

      const script = document.createElement('script');
      script.type = 'text/javascript';
      script.defer = true;
      script.src = this.scriptUrl;
      script.onload = () => {
        console.log('Zoho SalesIQ initialized');
        this.scriptLoaded = true;
        this.ensureNoChat();
      };

      document.body.appendChild(script);
    });

    return this.initPromise;
  }

  async setVisitorDetails(details: { name?: string; email?: string; SAN?: string; customFields?: any }): Promise<void> {
    await this.initialize(this.config);

    const zoho = (window as any).$zoho?.salesiq;
    if (!zoho) throw new Error('Zoho SalesIQ not initialized');

    try {
      if (details.name) zoho.visitor.name(details.name);
      if (details.email) zoho.visitor.email(details.email);

      // Track visitor details
      const enhancedInfo = {
        ...details.customFields,
        SAN: details.SAN,
        source: this.getAttributionData().source,
        campaign: this.getAttributionData().campaign,
        landing_page: window.location.href,
        referrer: document.referrer,
        first_visit: this.getFirstVisitDate(),
        ...this.getUtmParameters()
      };

      zoho.visitor.info(enhancedInfo);
      console.log('Visitor data sent successfully');
      this.ensureNoChat();
    } catch (error) {
      console.error('Error setting visitor details:', error);
      throw error;
    }
  }

  private ensureNoChat() {
    // Remove any existing chat UI
    const chatElements = document.querySelectorAll('[id^="zsiq_"], [class^="zsiq_"], [id^="zls"], [class^="zls"]');
    chatElements.forEach(el => el.remove());

    // Prevent future chat elements
    const style = document.createElement('style');
    style.textContent = `
      [id^="zsiq_"], [class^="zsiq_"],
      [id^="zls"], [class^="zls"],
      #zsiq_float, .zsiq_float, .zsiq_chatbtn, .zsiq_floatbtn {
        display: none !important;
        visibility: hidden !important;
        opacity: 0 !important;
        pointer-events: none !important;
      }
    `;
    document.head.appendChild(style);
  }

  unload() {
    console.log('Unloading Zoho SalesIQ');

    // Remove script
    const script = document.querySelector(`script[src="${this.scriptUrl}"]`);
    if (script) {
      script.remove();
      console.log('Zoho SalesIQ script removed');
    }

    // Clear global objects
    if ((window as any).$zoho) {
      delete (window as any).$zoho.salesiq;
    }

    // Remove chat UI elements
    this.ensureNoChat();

    // Reset script loaded state
    this.scriptLoaded = false;
    this.initPromise = null;
  }


  private setupAttributionTracking() {
    if (!localStorage.getItem('firstVisit')) {
      localStorage.setItem('firstVisit', new Date().toISOString());
    }
    if (!sessionStorage.getItem('initialReferrer') && document.referrer) {
      sessionStorage.setItem('initialReferrer', document.referrer);
    }
  }

  private getAttributionData() {
    return {
      source: sessionStorage.getItem('initialReferrer') || 'direct',
      campaign: this.getUtmParameters().utm_campaign || 'none'
    };
  }

  private getFirstVisitDate() {
    return localStorage.getItem('firstVisit') || new Date().toISOString();
  }

  private getUtmParameters() {
    const urlParams = new URLSearchParams(window.location.search);
    const utmParams: any = {};

    ['utm_source', 'utm_medium', 'utm_campaign', 'utm_content', 'utm_term'].forEach(param => {
      const value = urlParams.get(param);
      if (value) utmParams[param] = value;
    });

    return utmParams;
  }

  private handleWidgetReady() {
    this.ensureNoChat();
  }
}
