import { BehaviorSubject, from } from 'rxjs';
import { filter, first, map, switchMap } from 'rxjs/operators';
export class StripeInstance {
    constructor(version, loader, window, key, options) {
        this.version = version;
        this.loader = loader;
        this.window = window;
        this.key = key;
        this.options = options;
        this.stripe$ = new BehaviorSubject(undefined);
        this.stripe = this.stripe$
            .asObservable()
            .pipe(filter((stripe) => Boolean(stripe)));
        this.loader
            .asStream()
            .pipe(filter((status) => status.loaded === true), first(), map(() => this.window.getNativeWindow().Stripe))
            .subscribe((stripeInstance) => {
            const stripe = this.options
                ? stripeInstance(this.key, this.options)
                : stripeInstance(this.key);
            stripe.registerAppInfo(this.getNgxStripeAppInfo(this.version));
            this.stripe$.next(stripe);
        });
    }
    getInstance() {
        return this.stripe$.getValue();
    }
    elements(options) {
        return this.stripe$.asObservable().pipe(filter((stripe) => Boolean(stripe)), map((stripe) => stripe.elements(options)), first());
    }
    redirectToCheckout(options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.redirectToCheckout(options))), first());
    }
    confirmAuBecsDebitPayment(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmAuBecsDebitPayment(clientSecret, data))), first());
    }
    confirmBancontactPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmBancontactPayment(clientSecret, data, options))), first());
    }
    confirmCardPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmCardPayment(clientSecret, data, options))), first());
    }
    confirmEpsPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmEpsPayment(clientSecret, data, options))), first());
    }
    confirmFpxPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmFpxPayment(clientSecret, data, options))), first());
    }
    confirmGiropayPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmGiropayPayment(clientSecret, data, options))), first());
    }
    confirmIdealPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmIdealPayment(clientSecret, data, options))), first());
    }
    confirmP24Payment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmP24Payment(clientSecret, data, options))), first());
    }
    confirmSepaDebitPayment(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmSepaDebitPayment(clientSecret, data))), first());
    }
    handleCardAction(clientSecret) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleCardAction(clientSecret))), first());
    }
    createPaymentMethod(paymentMethodData) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.createPaymentMethod(paymentMethodData))), first());
    }
    retrievePaymentIntent(clientSecret) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.retrievePaymentIntent(clientSecret))), first());
    }
    confirmAuBecsDebitSetup(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmAuBecsDebitSetup(clientSecret, data))), first());
    }
    confirmCardSetup(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmCardSetup(clientSecret, data, options))), first());
    }
    confirmSepaDebitSetup(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmSepaDebitSetup(clientSecret, data))), first());
    }
    retrieveSetupIntent(clientSecret) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.retrieveSetupIntent(clientSecret))), first());
    }
    paymentRequest(options) {
        const stripe = this.getInstance();
        return stripe ? stripe.paymentRequest(options) : undefined;
    }
    createToken(tokenType, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.createToken(tokenType, data))), first());
    }
    createSource(a, b) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.createSource(a, b))), first());
    }
    retrieveSource(source) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.retrieveSource(source))), first());
    }
    /**
     * @deprecated
     */
    handleCardPayment(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleCardPayment(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    confirmPaymentIntent(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmPaymentIntent(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    handleCardSetup(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleCardSetup(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    confirmSetupIntent(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmSetupIntent(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    handleFpxPayment(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleFpxPayment(clientSecret, element, data))), first());
    }
    getNgxStripeAppInfo(version) {
        return {
            name: 'ngx-stripe',
            url: 'https://ngx-stripe.dev',
            partner_id: 'pp_partner_JR4l1rmvUoPP4V',
            version
        };
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RyaXBlLWluc3RhbmNlLmNsYXNzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vcHJvamVjdHMvbmd4LXN0cmlwZS9zcmMvbGliL3NlcnZpY2VzL3N0cmlwZS1pbnN0YW5jZS5jbGFzcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSxPQUFPLEVBQUUsZUFBZSxFQUFFLElBQUksRUFBYyxNQUFNLE1BQU0sQ0FBQztBQUN6RCxPQUFPLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsU0FBUyxFQUFFLE1BQU0sZ0JBQWdCLENBQUM7QUE0RC9ELE1BQU0sT0FBTyxjQUFjO0lBTXpCLFlBQ1UsT0FBZSxFQUNmLE1BQTJCLEVBQzNCLE1BQWlCLEVBQ2pCLEdBQVcsRUFDWCxPQUFrQztRQUpsQyxZQUFPLEdBQVAsT0FBTyxDQUFRO1FBQ2YsV0FBTSxHQUFOLE1BQU0sQ0FBcUI7UUFDM0IsV0FBTSxHQUFOLE1BQU0sQ0FBVztRQUNqQixRQUFHLEdBQUgsR0FBRyxDQUFRO1FBQ1gsWUFBTyxHQUFQLE9BQU8sQ0FBMkI7UUFWcEMsWUFBTyxHQUFHLElBQUksZUFBZSxDQUFxQixTQUFTLENBQUMsQ0FBQztRQUNyRSxXQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU87YUFDbEIsWUFBWSxFQUFFO2FBQ2QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQVMzQyxJQUFJLENBQUMsTUFBTTthQUNSLFFBQVEsRUFBRTthQUNWLElBQUksQ0FDSCxNQUFNLENBQUMsQ0FBQyxNQUFpQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxLQUFLLElBQUksQ0FBQyxFQUNyRSxLQUFLLEVBQUUsRUFDUCxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxlQUFlLEVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FDekQ7YUFDQSxTQUFTLENBQUMsQ0FBQyxjQUFtQixFQUFFLEVBQUU7WUFDakMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLE9BQU87Z0JBQ3pCLENBQUMsQ0FBRSxjQUFjLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFZO2dCQUNwRCxDQUFDLENBQUUsY0FBYyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQVksQ0FBQztZQUV4QyxNQUFjLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUN4RSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM1QixDQUFDLENBQUMsQ0FBQztJQUNQLENBQUM7SUFFRCxXQUFXO1FBQ1QsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxDQUFDO0lBQ2pDLENBQUM7SUFFRCxRQUFRLENBQUMsT0FBK0I7UUFDdEMsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxDQUFDLElBQUksQ0FDckMsTUFBTSxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsRUFDbkMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEVBQ3pDLEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQsa0JBQWtCLENBQ2hCLE9BQW1DO1FBRW5DLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQy9ELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQseUJBQXlCLENBQ3ZCLFlBQW9CLEVBQ3BCLElBQW9DO1FBS3BDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQ25CLElBQUksQ0FBQyxNQUFNLENBQUMseUJBQXlCLENBQUMsWUFBWSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQzNELEVBQ0QsS0FBSyxFQUFFLENBQ1IsQ0FBQztJQUNKLENBQUM7SUFFRCx3QkFBd0IsQ0FDdEIsWUFBb0IsRUFDcEIsSUFBbUMsRUFDbkMsT0FBeUM7UUFLekMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyx3QkFBd0IsQ0FBQyxZQUFZLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQ25FLEVBQ0QsS0FBSyxFQUFFLENBQ1IsQ0FBQztJQUNKLENBQUM7SUFFRCxrQkFBa0IsQ0FDaEIsWUFBb0IsRUFDcEIsSUFBNkIsRUFDN0IsT0FBbUM7UUFLbkMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxZQUFZLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQzdELEVBQ0QsS0FBSyxFQUFFLENBQ1IsQ0FBQztJQUNKLENBQUM7SUFFRCxpQkFBaUIsQ0FDZixZQUFvQixFQUNwQixJQUE0QixFQUM1QixPQUFrQztRQUtsQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUNuQixJQUFJLENBQUMsTUFBTSxDQUFDLGlCQUFpQixDQUFDLFlBQVksRUFBRSxJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FDNUQsRUFDRCxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVELGlCQUFpQixDQUNmLFlBQW9CLEVBQ3BCLElBQTRCLEVBQzVCLE9BQWtDO1FBS2xDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQ25CLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCLENBQUMsWUFBWSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUM1RCxFQUNELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQscUJBQXFCLENBQ25CLFlBQW9CLEVBQ3BCLElBQWdDLEVBQ2hDLE9BQXNDO1FBS3RDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQ25CLElBQUksQ0FBQyxNQUFNLENBQUMscUJBQXFCLENBQUMsWUFBWSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUNoRSxFQUNELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQsbUJBQW1CLENBQ2pCLFlBQW9CLEVBQ3BCLElBQThCLEVBQzlCLE9BQW9DO1FBS3BDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQ25CLElBQUksQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsWUFBWSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUM5RCxFQUNELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQsaUJBQWlCLENBQ2YsWUFBb0IsRUFDcEIsSUFBNEIsRUFDNUIsT0FBa0M7UUFLbEMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQzVELEVBQ0QsS0FBSyxFQUFFLENBQ1IsQ0FBQztJQUNKLENBQUM7SUFFRCx1QkFBdUIsQ0FDckIsWUFBb0IsRUFDcEIsSUFBa0M7UUFLbEMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyx1QkFBdUIsQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FDekQsRUFDRCxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVELGdCQUFnQixDQUNkLFlBQW9CO1FBS3BCLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLEVBQ2xFLEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQsbUJBQW1CLENBQ2pCLGlCQUEwQztRQUsxQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUNuQixJQUFJLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FDcEQsRUFDRCxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVELHFCQUFxQixDQUNuQixZQUFvQjtRQUtwQixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMscUJBQXFCLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxFQUN2RSxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVELHVCQUF1QixDQUNyQixZQUFvQixFQUNwQixJQUFrQztRQUtsQyxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUNuQixJQUFJLENBQUMsTUFBTSxDQUFDLHVCQUF1QixDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUN6RCxFQUNELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQsZ0JBQWdCLENBQ2QsWUFBb0IsRUFDcEIsSUFBMkIsRUFDM0IsT0FBaUM7UUFLakMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQzNELEVBQ0QsS0FBSyxFQUFFLENBQ1IsQ0FBQztJQUNKLENBQUM7SUFFRCxxQkFBcUIsQ0FDbkIsWUFBb0IsRUFDcEIsSUFBZ0M7UUFLaEMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDbkIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FDdkQsRUFDRCxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVELG1CQUFtQixDQUNqQixZQUFvQjtRQUtwQixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsbUJBQW1CLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxFQUNyRSxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVELGNBQWMsQ0FBQyxPQUE4QjtRQUMzQyxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7UUFFbEMsT0FBTyxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUM3RCxDQUFDO0lBOEJELFdBQVcsQ0FBQyxTQUFTLEVBQUUsSUFBSTtRQUN6QixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQ2hFLEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBU0QsWUFBWSxDQUFDLENBQUMsRUFBRSxDQUFFO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFDdEQsS0FBSyxFQUFFLENBQ1IsQ0FBQztJQUNKLENBQUM7SUFFRCxjQUFjLENBQ1osTUFBMkI7UUFFM0IsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQzFELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSCxpQkFBaUIsQ0FDZixZQUFvQixFQUNwQixPQUFRLEVBQ1IsSUFBSztRQUtMLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQ3JCLFNBQVMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQ25CLElBQUksQ0FBRSxNQUFjLENBQUMsaUJBQWlCLENBQUMsWUFBWSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUNyRSxFQUNELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSCxvQkFBb0IsQ0FDbEIsWUFBb0IsRUFDcEIsT0FBUSxFQUNSLElBQUs7UUFLTCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUNuQixJQUFJLENBQUUsTUFBYyxDQUFDLG9CQUFvQixDQUFDLFlBQVksRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FDeEUsRUFDRCxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0gsZUFBZSxDQUNiLFlBQW9CLEVBQ3BCLE9BQVEsRUFDUixJQUFLO1FBS0wsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FDckIsU0FBUyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDbkIsSUFBSSxDQUFFLE1BQWMsQ0FBQyxlQUFlLENBQUMsWUFBWSxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUNuRSxFQUNELEtBQUssRUFBRSxDQUNSLENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSCxrQkFBa0IsQ0FDaEIsWUFBb0IsRUFDcEIsT0FBUSxFQUNSLElBQUs7UUFLTCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUNuQixJQUFJLENBQUUsTUFBYyxDQUFDLGtCQUFrQixDQUFDLFlBQVksRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FDdEUsRUFDRCxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0gsZ0JBQWdCLENBQ2QsWUFBb0IsRUFDcEIsT0FBUSxFQUNSLElBQUs7UUFLTCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUNyQixTQUFTLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRSxDQUNuQixJQUFJLENBQUUsTUFBYyxDQUFDLGdCQUFnQixDQUFDLFlBQVksRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FDcEUsRUFDRCxLQUFLLEVBQUUsQ0FDUixDQUFDO0lBQ0osQ0FBQztJQUVPLG1CQUFtQixDQUFDLE9BQWU7UUFDekMsT0FBTztZQUNMLElBQUksRUFBRSxZQUFZO1lBQ2xCLEdBQUcsRUFBRSx3QkFBd0I7WUFDN0IsVUFBVSxFQUFFLDJCQUEyQjtZQUN2QyxPQUFPO1NBQ1IsQ0FBQztJQUNKLENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEJlaGF2aW9yU3ViamVjdCwgZnJvbSwgT2JzZXJ2YWJsZSB9IGZyb20gJ3J4anMnO1xuaW1wb3J0IHsgZmlsdGVyLCBmaXJzdCwgbWFwLCBzd2l0Y2hNYXAgfSBmcm9tICdyeGpzL29wZXJhdG9ycyc7XG5cbmltcG9ydCB7XG4gIENvbmZpcm1BdUJlY3NEZWJpdFBheW1lbnREYXRhLFxuICBDb25maXJtQXVCZWNzRGViaXRTZXR1cERhdGEsXG4gIENvbmZpcm1CYW5jb250YWN0UGF5bWVudERhdGEsXG4gIENvbmZpcm1CYW5jb250YWN0UGF5bWVudE9wdGlvbnMsXG4gIENvbmZpcm1DYXJkUGF5bWVudERhdGEsXG4gIENvbmZpcm1DYXJkUGF5bWVudE9wdGlvbnMsXG4gIENvbmZpcm1FcHNQYXltZW50RGF0YSxcbiAgQ29uZmlybUVwc1BheW1lbnRPcHRpb25zLFxuICBDb25maXJtRnB4UGF5bWVudERhdGEsXG4gIENvbmZpcm1GcHhQYXltZW50T3B0aW9ucyxcbiAgQ29uZmlybUdpcm9wYXlQYXltZW50RGF0YSxcbiAgQ29uZmlybUdpcm9wYXlQYXltZW50T3B0aW9ucyxcbiAgQ29uZmlybUlkZWFsUGF5bWVudERhdGEsXG4gIENvbmZpcm1JZGVhbFBheW1lbnRPcHRpb25zLFxuICBDb25maXJtUDI0UGF5bWVudERhdGEsXG4gIENvbmZpcm1QMjRQYXltZW50T3B0aW9ucyxcbiAgQ29uZmlybUNhcmRTZXR1cERhdGEsXG4gIENvbmZpcm1DYXJkU2V0dXBPcHRpb25zLFxuICBDb25maXJtU2VwYURlYml0UGF5bWVudERhdGEsXG4gIENvbmZpcm1TZXBhRGViaXRTZXR1cERhdGEsXG4gIENyZWF0ZVBheW1lbnRNZXRob2REYXRhLFxuICBDcmVhdGVTb3VyY2VEYXRhLFxuICBDcmVhdGVUb2tlbkliYW5EYXRhLFxuICBDcmVhdGVUb2tlbkNhcmREYXRhLFxuICBDcmVhdGVUb2tlblBpaURhdGEsXG4gIENyZWF0ZVRva2VuQmFua0FjY291bnREYXRhLFxuICBQYXltZW50SW50ZW50LFxuICBQYXltZW50TWV0aG9kLFxuICBQYXltZW50UmVxdWVzdCxcbiAgUGF5bWVudFJlcXVlc3RPcHRpb25zLFxuICBSZWRpcmVjdFRvQ2hlY2tvdXRPcHRpb25zLFxuICBSZXRyaWV2ZVNvdXJjZVBhcmFtLFxuICBTZXR1cEludGVudCxcbiAgU3RyaXBlLFxuICBTdHJpcGVDYXJkRWxlbWVudCxcbiAgU3RyaXBlQ2FyZE51bWJlckVsZW1lbnQsXG4gIFN0cmlwZUNhcmRDdmNFbGVtZW50LFxuICBTdHJpcGVDb25zdHJ1Y3Rvck9wdGlvbnMsXG4gIFN0cmlwZUVsZW1lbnRzLFxuICBTdHJpcGVFbGVtZW50c09wdGlvbnMsXG4gIFN0cmlwZUVsZW1lbnQsXG4gIFN0cmlwZUVycm9yLFxuICBTdHJpcGVJYmFuRWxlbWVudCxcbiAgU291cmNlLFxuICBUb2tlbixcbiAgVG9rZW5DcmVhdGVQYXJhbXNcbn0gZnJvbSAnQHN0cmlwZS9zdHJpcGUtanMnO1xuXG5pbXBvcnQgeyBTdHJpcGVBcHBJbmZvIH0gZnJvbSAnLi4vaW50ZXJmYWNlcy9uZ3gtc3RyaXBlLmludGVyZmFjZSc7XG5pbXBvcnQgeyBTdHJpcGVTZXJ2aWNlSW50ZXJmYWNlIH0gZnJvbSAnLi4vaW50ZXJmYWNlcy9zdHJpcGUtaW5zdGFuY2UuaW50ZXJmYWNlJztcblxuaW1wb3J0IHsgV2luZG93UmVmIH0gZnJvbSAnLi93aW5kb3ctcmVmLnNlcnZpY2UnO1xuaW1wb3J0IHtcbiAgTGF6eVN0cmlwZUFQSUxvYWRlcixcbiAgTGF6eVN0cmlwZUFQSUxvYWRlclN0YXR1c1xufSBmcm9tICcuL2FwaS1sb2FkZXIuc2VydmljZSc7XG5cbmV4cG9ydCBjbGFzcyBTdHJpcGVJbnN0YW5jZSBpbXBsZW1lbnRzIFN0cmlwZVNlcnZpY2VJbnRlcmZhY2Uge1xuICBwcml2YXRlIHN0cmlwZSQgPSBuZXcgQmVoYXZpb3JTdWJqZWN0PFN0cmlwZSB8IHVuZGVmaW5lZD4odW5kZWZpbmVkKTtcbiAgc3RyaXBlID0gdGhpcy5zdHJpcGUkXG4gICAgLmFzT2JzZXJ2YWJsZSgpXG4gICAgLnBpcGUoZmlsdGVyKChzdHJpcGUpID0+IEJvb2xlYW4oc3RyaXBlKSkpO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHByaXZhdGUgdmVyc2lvbjogc3RyaW5nLFxuICAgIHByaXZhdGUgbG9hZGVyOiBMYXp5U3RyaXBlQVBJTG9hZGVyLFxuICAgIHByaXZhdGUgd2luZG93OiBXaW5kb3dSZWYsXG4gICAgcHJpdmF0ZSBrZXk6IHN0cmluZyxcbiAgICBwcml2YXRlIG9wdGlvbnM/OiBTdHJpcGVDb25zdHJ1Y3Rvck9wdGlvbnNcbiAgKSB7XG4gICAgdGhpcy5sb2FkZXJcbiAgICAgIC5hc1N0cmVhbSgpXG4gICAgICAucGlwZShcbiAgICAgICAgZmlsdGVyKChzdGF0dXM6IExhenlTdHJpcGVBUElMb2FkZXJTdGF0dXMpID0+IHN0YXR1cy5sb2FkZWQgPT09IHRydWUpLFxuICAgICAgICBmaXJzdCgpLFxuICAgICAgICBtYXAoKCkgPT4gKHRoaXMud2luZG93LmdldE5hdGl2ZVdpbmRvdygpIGFzIGFueSkuU3RyaXBlKVxuICAgICAgKVxuICAgICAgLnN1YnNjcmliZSgoc3RyaXBlSW5zdGFuY2U6IGFueSkgPT4ge1xuICAgICAgICBjb25zdCBzdHJpcGUgPSB0aGlzLm9wdGlvbnNcbiAgICAgICAgICA/IChzdHJpcGVJbnN0YW5jZSh0aGlzLmtleSwgdGhpcy5vcHRpb25zKSBhcyBTdHJpcGUpXG4gICAgICAgICAgOiAoc3RyaXBlSW5zdGFuY2UodGhpcy5rZXkpIGFzIFN0cmlwZSk7XG5cbiAgICAgICAgKHN0cmlwZSBhcyBhbnkpLnJlZ2lzdGVyQXBwSW5mbyh0aGlzLmdldE5neFN0cmlwZUFwcEluZm8odGhpcy52ZXJzaW9uKSk7XG4gICAgICAgIHRoaXMuc3RyaXBlJC5uZXh0KHN0cmlwZSk7XG4gICAgICB9KTtcbiAgfVxuXG4gIGdldEluc3RhbmNlKCk6IFN0cmlwZSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlJC5nZXRWYWx1ZSgpO1xuICB9XG5cbiAgZWxlbWVudHMob3B0aW9ucz86IFN0cmlwZUVsZW1lbnRzT3B0aW9ucyk6IE9ic2VydmFibGU8U3RyaXBlRWxlbWVudHM+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUkLmFzT2JzZXJ2YWJsZSgpLnBpcGUoXG4gICAgICBmaWx0ZXIoKHN0cmlwZSkgPT4gQm9vbGVhbihzdHJpcGUpKSxcbiAgICAgIG1hcCgoc3RyaXBlKSA9PiBzdHJpcGUuZWxlbWVudHMob3B0aW9ucykpLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICByZWRpcmVjdFRvQ2hlY2tvdXQoXG4gICAgb3B0aW9ucz86IFJlZGlyZWN0VG9DaGVja291dE9wdGlvbnNcbiAgKTogT2JzZXJ2YWJsZTxuZXZlciB8IHsgZXJyb3I6IFN0cmlwZUVycm9yIH0+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUucGlwZShcbiAgICAgIHN3aXRjaE1hcCgoc3RyaXBlKSA9PiBmcm9tKHN0cmlwZS5yZWRpcmVjdFRvQ2hlY2tvdXQob3B0aW9ucykpKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgY29uZmlybUF1QmVjc0RlYml0UGF5bWVudChcbiAgICBjbGllbnRTZWNyZXQ6IHN0cmluZyxcbiAgICBkYXRhPzogQ29uZmlybUF1QmVjc0RlYml0UGF5bWVudERhdGFcbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgcGF5bWVudEludGVudD86IFBheW1lbnRJbnRlbnQ7XG4gICAgZXJyb3I/OiBTdHJpcGVFcnJvcjtcbiAgfT4ge1xuICAgIHJldHVybiB0aGlzLnN0cmlwZS5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzdHJpcGUpID0+XG4gICAgICAgIGZyb20oc3RyaXBlLmNvbmZpcm1BdUJlY3NEZWJpdFBheW1lbnQoY2xpZW50U2VjcmV0LCBkYXRhKSlcbiAgICAgICksXG4gICAgICBmaXJzdCgpXG4gICAgKTtcbiAgfVxuXG4gIGNvbmZpcm1CYW5jb250YWN0UGF5bWVudChcbiAgICBjbGllbnRTZWNyZXQ6IHN0cmluZyxcbiAgICBkYXRhPzogQ29uZmlybUJhbmNvbnRhY3RQYXltZW50RGF0YSxcbiAgICBvcHRpb25zPzogQ29uZmlybUJhbmNvbnRhY3RQYXltZW50T3B0aW9uc1xuICApOiBPYnNlcnZhYmxlPHtcbiAgICBwYXltZW50SW50ZW50PzogUGF5bWVudEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbShzdHJpcGUuY29uZmlybUJhbmNvbnRhY3RQYXltZW50KGNsaWVudFNlY3JldCwgZGF0YSwgb3B0aW9ucykpXG4gICAgICApLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICBjb25maXJtQ2FyZFBheW1lbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1DYXJkUGF5bWVudERhdGEsXG4gICAgb3B0aW9ucz86IENvbmZpcm1DYXJkUGF5bWVudE9wdGlvbnNcbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgcGF5bWVudEludGVudD86IFBheW1lbnRJbnRlbnQ7XG4gICAgZXJyb3I/OiBTdHJpcGVFcnJvcjtcbiAgfT4ge1xuICAgIHJldHVybiB0aGlzLnN0cmlwZS5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzdHJpcGUpID0+XG4gICAgICAgIGZyb20oc3RyaXBlLmNvbmZpcm1DYXJkUGF5bWVudChjbGllbnRTZWNyZXQsIGRhdGEsIG9wdGlvbnMpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgY29uZmlybUVwc1BheW1lbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1FcHNQYXltZW50RGF0YSxcbiAgICBvcHRpb25zPzogQ29uZmlybUVwc1BheW1lbnRPcHRpb25zXG4gICk6IE9ic2VydmFibGU8e1xuICAgIHBheW1lbnRJbnRlbnQ/OiBQYXltZW50SW50ZW50O1xuICAgIGVycm9yPzogU3RyaXBlRXJyb3I7XG4gIH0+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUucGlwZShcbiAgICAgIHN3aXRjaE1hcCgoc3RyaXBlKSA9PlxuICAgICAgICBmcm9tKHN0cmlwZS5jb25maXJtRXBzUGF5bWVudChjbGllbnRTZWNyZXQsIGRhdGEsIG9wdGlvbnMpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgY29uZmlybUZweFBheW1lbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1GcHhQYXltZW50RGF0YSxcbiAgICBvcHRpb25zPzogQ29uZmlybUZweFBheW1lbnRPcHRpb25zXG4gICk6IE9ic2VydmFibGU8e1xuICAgIHBheW1lbnRJbnRlbnQ/OiBQYXltZW50SW50ZW50O1xuICAgIGVycm9yPzogU3RyaXBlRXJyb3I7XG4gIH0+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUucGlwZShcbiAgICAgIHN3aXRjaE1hcCgoc3RyaXBlKSA9PlxuICAgICAgICBmcm9tKHN0cmlwZS5jb25maXJtRnB4UGF5bWVudChjbGllbnRTZWNyZXQsIGRhdGEsIG9wdGlvbnMpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgY29uZmlybUdpcm9wYXlQYXltZW50KFxuICAgIGNsaWVudFNlY3JldDogc3RyaW5nLFxuICAgIGRhdGE/OiBDb25maXJtR2lyb3BheVBheW1lbnREYXRhLFxuICAgIG9wdGlvbnM/OiBDb25maXJtR2lyb3BheVBheW1lbnRPcHRpb25zXG4gICk6IE9ic2VydmFibGU8e1xuICAgIHBheW1lbnRJbnRlbnQ/OiBQYXltZW50SW50ZW50O1xuICAgIGVycm9yPzogU3RyaXBlRXJyb3I7XG4gIH0+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUucGlwZShcbiAgICAgIHN3aXRjaE1hcCgoc3RyaXBlKSA9PlxuICAgICAgICBmcm9tKHN0cmlwZS5jb25maXJtR2lyb3BheVBheW1lbnQoY2xpZW50U2VjcmV0LCBkYXRhLCBvcHRpb25zKSlcbiAgICAgICksXG4gICAgICBmaXJzdCgpXG4gICAgKTtcbiAgfVxuXG4gIGNvbmZpcm1JZGVhbFBheW1lbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1JZGVhbFBheW1lbnREYXRhLFxuICAgIG9wdGlvbnM/OiBDb25maXJtSWRlYWxQYXltZW50T3B0aW9uc1xuICApOiBPYnNlcnZhYmxlPHtcbiAgICBwYXltZW50SW50ZW50PzogUGF5bWVudEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbShzdHJpcGUuY29uZmlybUlkZWFsUGF5bWVudChjbGllbnRTZWNyZXQsIGRhdGEsIG9wdGlvbnMpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgY29uZmlybVAyNFBheW1lbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1QMjRQYXltZW50RGF0YSxcbiAgICBvcHRpb25zPzogQ29uZmlybVAyNFBheW1lbnRPcHRpb25zXG4gICk6IE9ic2VydmFibGU8e1xuICAgIHBheW1lbnRJbnRlbnQ/OiBQYXltZW50SW50ZW50O1xuICAgIGVycm9yPzogU3RyaXBlRXJyb3I7XG4gIH0+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUucGlwZShcbiAgICAgIHN3aXRjaE1hcCgoc3RyaXBlKSA9PlxuICAgICAgICBmcm9tKHN0cmlwZS5jb25maXJtUDI0UGF5bWVudChjbGllbnRTZWNyZXQsIGRhdGEsIG9wdGlvbnMpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgY29uZmlybVNlcGFEZWJpdFBheW1lbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1TZXBhRGViaXRQYXltZW50RGF0YVxuICApOiBPYnNlcnZhYmxlPHtcbiAgICBwYXltZW50SW50ZW50PzogUGF5bWVudEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbShzdHJpcGUuY29uZmlybVNlcGFEZWJpdFBheW1lbnQoY2xpZW50U2VjcmV0LCBkYXRhKSlcbiAgICAgICksXG4gICAgICBmaXJzdCgpXG4gICAgKTtcbiAgfVxuXG4gIGhhbmRsZUNhcmRBY3Rpb24oXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmdcbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgcGF5bWVudEludGVudD86IFBheW1lbnRJbnRlbnQ7XG4gICAgZXJyb3I/OiBTdHJpcGVFcnJvcjtcbiAgfT4ge1xuICAgIHJldHVybiB0aGlzLnN0cmlwZS5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzdHJpcGUpID0+IGZyb20oc3RyaXBlLmhhbmRsZUNhcmRBY3Rpb24oY2xpZW50U2VjcmV0KSkpLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICBjcmVhdGVQYXltZW50TWV0aG9kKFxuICAgIHBheW1lbnRNZXRob2REYXRhOiBDcmVhdGVQYXltZW50TWV0aG9kRGF0YVxuICApOiBPYnNlcnZhYmxlPHtcbiAgICBwYXltZW50TWV0aG9kPzogUGF5bWVudE1ldGhvZDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbShzdHJpcGUuY3JlYXRlUGF5bWVudE1ldGhvZChwYXltZW50TWV0aG9kRGF0YSkpXG4gICAgICApLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICByZXRyaWV2ZVBheW1lbnRJbnRlbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmdcbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgcGF5bWVudEludGVudD86IFBheW1lbnRJbnRlbnQ7XG4gICAgZXJyb3I/OiBTdHJpcGVFcnJvcjtcbiAgfT4ge1xuICAgIHJldHVybiB0aGlzLnN0cmlwZS5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzdHJpcGUpID0+IGZyb20oc3RyaXBlLnJldHJpZXZlUGF5bWVudEludGVudChjbGllbnRTZWNyZXQpKSksXG4gICAgICBmaXJzdCgpXG4gICAgKTtcbiAgfVxuXG4gIGNvbmZpcm1BdUJlY3NEZWJpdFNldHVwKFxuICAgIGNsaWVudFNlY3JldDogc3RyaW5nLFxuICAgIGRhdGE/OiBDb25maXJtQXVCZWNzRGViaXRTZXR1cERhdGFcbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgc2V0dXBJbnRlbnQ/OiBTZXR1cEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbShzdHJpcGUuY29uZmlybUF1QmVjc0RlYml0U2V0dXAoY2xpZW50U2VjcmV0LCBkYXRhKSlcbiAgICAgICksXG4gICAgICBmaXJzdCgpXG4gICAgKTtcbiAgfVxuXG4gIGNvbmZpcm1DYXJkU2V0dXAoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1DYXJkU2V0dXBEYXRhLFxuICAgIG9wdGlvbnM/OiBDb25maXJtQ2FyZFNldHVwT3B0aW9uc1xuICApOiBPYnNlcnZhYmxlPHtcbiAgICBzZXR1cEludGVudD86IFNldHVwSW50ZW50O1xuICAgIGVycm9yPzogU3RyaXBlRXJyb3I7XG4gIH0+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUucGlwZShcbiAgICAgIHN3aXRjaE1hcCgoc3RyaXBlKSA9PlxuICAgICAgICBmcm9tKHN0cmlwZS5jb25maXJtQ2FyZFNldHVwKGNsaWVudFNlY3JldCwgZGF0YSwgb3B0aW9ucykpXG4gICAgICApLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICBjb25maXJtU2VwYURlYml0U2V0dXAoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZGF0YT86IENvbmZpcm1TZXBhRGViaXRTZXR1cERhdGFcbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgc2V0dXBJbnRlbnQ/OiBTZXR1cEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbShzdHJpcGUuY29uZmlybVNlcGFEZWJpdFNldHVwKGNsaWVudFNlY3JldCwgZGF0YSkpXG4gICAgICApLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICByZXRyaWV2ZVNldHVwSW50ZW50KFxuICAgIGNsaWVudFNlY3JldDogc3RyaW5nXG4gICk6IE9ic2VydmFibGU8e1xuICAgIHNldHVwSW50ZW50PzogU2V0dXBJbnRlbnQ7XG4gICAgZXJyb3I/OiBTdHJpcGVFcnJvcjtcbiAgfT4ge1xuICAgIHJldHVybiB0aGlzLnN0cmlwZS5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzdHJpcGUpID0+IGZyb20oc3RyaXBlLnJldHJpZXZlU2V0dXBJbnRlbnQoY2xpZW50U2VjcmV0KSkpLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICBwYXltZW50UmVxdWVzdChvcHRpb25zOiBQYXltZW50UmVxdWVzdE9wdGlvbnMpOiBQYXltZW50UmVxdWVzdCB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3Qgc3RyaXBlID0gdGhpcy5nZXRJbnN0YW5jZSgpO1xuXG4gICAgcmV0dXJuIHN0cmlwZSA/IHN0cmlwZS5wYXltZW50UmVxdWVzdChvcHRpb25zKSA6IHVuZGVmaW5lZDtcbiAgfVxuXG4gIGNyZWF0ZVRva2VuKFxuICAgIHRva2VuVHlwZTogU3RyaXBlSWJhbkVsZW1lbnQsXG4gICAgZGF0YTogQ3JlYXRlVG9rZW5JYmFuRGF0YVxuICApOiBPYnNlcnZhYmxlPHsgdG9rZW4/OiBUb2tlbjsgZXJyb3I/OiBTdHJpcGVFcnJvciB9PjtcbiAgY3JlYXRlVG9rZW4oXG4gICAgdG9rZW5UeXBlOiBTdHJpcGVDYXJkRWxlbWVudCB8IFN0cmlwZUNhcmROdW1iZXJFbGVtZW50LFxuICAgIGRhdGE/OiBDcmVhdGVUb2tlbkNhcmREYXRhXG4gICk6IE9ic2VydmFibGU8eyB0b2tlbj86IFRva2VuOyBlcnJvcj86IFN0cmlwZUVycm9yIH0+O1xuICBjcmVhdGVUb2tlbihcbiAgICB0b2tlblR5cGU6ICdwaWknLFxuICAgIGRhdGE6IENyZWF0ZVRva2VuUGlpRGF0YVxuICApOiBPYnNlcnZhYmxlPHsgdG9rZW4/OiBUb2tlbjsgZXJyb3I/OiBTdHJpcGVFcnJvciB9PjtcbiAgY3JlYXRlVG9rZW4oXG4gICAgdG9rZW5UeXBlOiAnYmFua19hY2NvdW50JyxcbiAgICBkYXRhOiBDcmVhdGVUb2tlbkJhbmtBY2NvdW50RGF0YVxuICApOiBPYnNlcnZhYmxlPHsgdG9rZW4/OiBUb2tlbjsgZXJyb3I/OiBTdHJpcGVFcnJvciB9PjtcbiAgY3JlYXRlVG9rZW4oXG4gICAgdG9rZW5UeXBlOiAnY3ZjX3VwZGF0ZScsXG4gICAgZWxlbWVudD86IFN0cmlwZUNhcmRDdmNFbGVtZW50XG4gICk6IE9ic2VydmFibGU8eyB0b2tlbj86IFRva2VuOyBlcnJvcj86IFN0cmlwZUVycm9yIH0+O1xuICBjcmVhdGVUb2tlbihcbiAgICB0b2tlblR5cGU6ICdhY2NvdW50JyxcbiAgICBkYXRhOiBUb2tlbkNyZWF0ZVBhcmFtcy5BY2NvdW50XG4gICk6IE9ic2VydmFibGU8eyB0b2tlbj86IFRva2VuOyBlcnJvcj86IFN0cmlwZUVycm9yIH0+O1xuICBjcmVhdGVUb2tlbihcbiAgICB0b2tlblR5cGU6ICdwZXJzb24nLFxuICAgIGRhdGE6IFRva2VuQ3JlYXRlUGFyYW1zLlBlcnNvblxuICApOiBPYnNlcnZhYmxlPHsgdG9rZW4/OiBUb2tlbjsgZXJyb3I/OiBTdHJpcGVFcnJvciB9PjtcbiAgY3JlYXRlVG9rZW4odG9rZW5UeXBlLCBkYXRhKSB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT4gZnJvbShzdHJpcGUuY3JlYXRlVG9rZW4odG9rZW5UeXBlLCBkYXRhKSkpLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICBjcmVhdGVTb3VyY2UoXG4gICAgZWxlbWVudDogU3RyaXBlRWxlbWVudCxcbiAgICBzb3VyY2VEYXRhOiBDcmVhdGVTb3VyY2VEYXRhXG4gICk6IE9ic2VydmFibGU8eyBzb3VyY2U/OiBTb3VyY2U7IGVycm9yPzogU3RyaXBlRXJyb3IgfT47XG4gIGNyZWF0ZVNvdXJjZShcbiAgICBzb3VyY2VEYXRhOiBDcmVhdGVTb3VyY2VEYXRhXG4gICk6IE9ic2VydmFibGU8eyBzb3VyY2U/OiBTb3VyY2U7IGVycm9yPzogU3RyaXBlRXJyb3IgfT47XG4gIGNyZWF0ZVNvdXJjZShhLCBiPyk6IE9ic2VydmFibGU8eyBzb3VyY2U/OiBTb3VyY2U7IGVycm9yPzogU3RyaXBlRXJyb3IgfT4ge1xuICAgIHJldHVybiB0aGlzLnN0cmlwZS5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzdHJpcGUpID0+IGZyb20oc3RyaXBlLmNyZWF0ZVNvdXJjZShhLCBiKSkpLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICByZXRyaWV2ZVNvdXJjZShcbiAgICBzb3VyY2U6IFJldHJpZXZlU291cmNlUGFyYW1cbiAgKTogT2JzZXJ2YWJsZTx7IHNvdXJjZT86IFNvdXJjZTsgZXJyb3I/OiBTdHJpcGVFcnJvciB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT4gZnJvbShzdHJpcGUucmV0cmlldmVTb3VyY2Uoc291cmNlKSkpLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWRcbiAgICovXG4gIGhhbmRsZUNhcmRQYXltZW50KFxuICAgIGNsaWVudFNlY3JldDogc3RyaW5nLFxuICAgIGVsZW1lbnQ/LFxuICAgIGRhdGE/XG4gICk6IE9ic2VydmFibGU8e1xuICAgIHBheW1lbnRJbnRlbnQ/OiBQYXltZW50SW50ZW50O1xuICAgIGVycm9yPzogU3RyaXBlRXJyb3I7XG4gIH0+IHtcbiAgICByZXR1cm4gdGhpcy5zdHJpcGUucGlwZShcbiAgICAgIHN3aXRjaE1hcCgoc3RyaXBlKSA9PlxuICAgICAgICBmcm9tKChzdHJpcGUgYXMgYW55KS5oYW5kbGVDYXJkUGF5bWVudChjbGllbnRTZWNyZXQsIGVsZW1lbnQsIGRhdGEpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkXG4gICAqL1xuICBjb25maXJtUGF5bWVudEludGVudChcbiAgICBjbGllbnRTZWNyZXQ6IHN0cmluZyxcbiAgICBlbGVtZW50PyxcbiAgICBkYXRhP1xuICApOiBPYnNlcnZhYmxlPHtcbiAgICBwYXltZW50SW50ZW50PzogUGF5bWVudEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbSgoc3RyaXBlIGFzIGFueSkuY29uZmlybVBheW1lbnRJbnRlbnQoY2xpZW50U2VjcmV0LCBlbGVtZW50LCBkYXRhKSlcbiAgICAgICksXG4gICAgICBmaXJzdCgpXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAZGVwcmVjYXRlZFxuICAgKi9cbiAgaGFuZGxlQ2FyZFNldHVwKFxuICAgIGNsaWVudFNlY3JldDogc3RyaW5nLFxuICAgIGVsZW1lbnQ/LFxuICAgIGRhdGE/XG4gICk6IE9ic2VydmFibGU8e1xuICAgIHNldHVwSW50ZW50PzogU2V0dXBJbnRlbnQ7XG4gICAgZXJyb3I/OiBTdHJpcGVFcnJvcjtcbiAgfT4ge1xuICAgIHJldHVybiB0aGlzLnN0cmlwZS5waXBlKFxuICAgICAgc3dpdGNoTWFwKChzdHJpcGUpID0+XG4gICAgICAgIGZyb20oKHN0cmlwZSBhcyBhbnkpLmhhbmRsZUNhcmRTZXR1cChjbGllbnRTZWNyZXQsIGVsZW1lbnQsIGRhdGEpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkXG4gICAqL1xuICBjb25maXJtU2V0dXBJbnRlbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZWxlbWVudD8sXG4gICAgZGF0YT9cbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgc2V0dXBJbnRlbnQ/OiBTZXR1cEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbSgoc3RyaXBlIGFzIGFueSkuY29uZmlybVNldHVwSW50ZW50KGNsaWVudFNlY3JldCwgZWxlbWVudCwgZGF0YSkpXG4gICAgICApLFxuICAgICAgZmlyc3QoKVxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWRcbiAgICovXG4gIGhhbmRsZUZweFBheW1lbnQoXG4gICAgY2xpZW50U2VjcmV0OiBzdHJpbmcsXG4gICAgZWxlbWVudD8sXG4gICAgZGF0YT9cbiAgKTogT2JzZXJ2YWJsZTx7XG4gICAgc2V0dXBJbnRlbnQ/OiBTZXR1cEludGVudDtcbiAgICBlcnJvcj86IFN0cmlwZUVycm9yO1xuICB9PiB7XG4gICAgcmV0dXJuIHRoaXMuc3RyaXBlLnBpcGUoXG4gICAgICBzd2l0Y2hNYXAoKHN0cmlwZSkgPT5cbiAgICAgICAgZnJvbSgoc3RyaXBlIGFzIGFueSkuaGFuZGxlRnB4UGF5bWVudChjbGllbnRTZWNyZXQsIGVsZW1lbnQsIGRhdGEpKVxuICAgICAgKSxcbiAgICAgIGZpcnN0KClcbiAgICApO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXROZ3hTdHJpcGVBcHBJbmZvKHZlcnNpb246IHN0cmluZyk6IFN0cmlwZUFwcEluZm8ge1xuICAgIHJldHVybiB7XG4gICAgICBuYW1lOiAnbmd4LXN0cmlwZScsXG4gICAgICB1cmw6ICdodHRwczovL25neC1zdHJpcGUuZGV2JyxcbiAgICAgIHBhcnRuZXJfaWQ6ICdwcF9wYXJ0bmVyX0pSNGwxcm12VW9QUDRWJyxcbiAgICAgIHZlcnNpb25cbiAgICB9O1xuICB9XG59XG4iXX0=