import { Injectable, Inject } from '@angular/core';
import { filter, map } from 'rxjs/operators';
import { STRIPE_PUBLISHABLE_KEY, STRIPE_OPTIONS, NGX_STRIPE_VERSION } from '../interfaces/ngx-stripe.interface';
import { WindowRef } from './window-ref.service';
import { LazyStripeAPILoader } from './api-loader.service';
import { StripeInstance } from './stripe-instance.class';
export class StripeService {
    constructor(version, key, options, loader, window) {
        this.version = version;
        this.key = key;
        this.options = options;
        this.loader = loader;
        this.window = window;
        if (key) {
            this.stripe = new StripeInstance(this.version, this.loader, this.window, key, options);
        }
    }
    getStripeReference() {
        return this.loader.asStream().pipe(filter((status) => status.loaded === true), map(() => this.window.getNativeWindow().Stripe));
    }
    getInstance() {
        return this.stripe.getInstance();
    }
    setKey(key, options) {
        return this.changeKey(key, options);
    }
    changeKey(key, options) {
        this.stripe = new StripeInstance(this.version, this.loader, this.window, key, options);
        return this.stripe;
    }
    elements(options) {
        return this.stripe.elements(options);
    }
    redirectToCheckout(options) {
        return this.stripe.redirectToCheckout(options);
    }
    confirmAuBecsDebitPayment(clientSecret, data) {
        return this.stripe.confirmAuBecsDebitPayment(clientSecret, data);
    }
    confirmBancontactPayment(clientSecret, data, options) {
        return this.stripe.confirmBancontactPayment(clientSecret, data, options);
    }
    confirmCardPayment(clientSecret, data, options) {
        return this.stripe.confirmCardPayment(clientSecret, data, options);
    }
    confirmEpsPayment(clientSecret, data, options) {
        return this.stripe.confirmEpsPayment(clientSecret, data, options);
    }
    confirmFpxPayment(clientSecret, data, options) {
        return this.stripe.confirmFpxPayment(clientSecret, data, options);
    }
    confirmGiropayPayment(clientSecret, data, options) {
        return this.stripe.confirmGiropayPayment(clientSecret, data, options);
    }
    confirmIdealPayment(clientSecret, data, options) {
        return this.stripe.confirmIdealPayment(clientSecret, data, options);
    }
    confirmP24Payment(clientSecret, data, options) {
        return this.stripe.confirmP24Payment(clientSecret, data, options);
    }
    confirmSepaDebitPayment(clientSecret, data) {
        return this.stripe.confirmSepaDebitPayment(clientSecret, data);
    }
    handleCardAction(clientSecret) {
        return this.stripe.handleCardAction(clientSecret);
    }
    createPaymentMethod(paymentMethodData) {
        return this.stripe.createPaymentMethod(paymentMethodData);
    }
    retrievePaymentIntent(clientSecret) {
        return this.stripe.retrievePaymentIntent(clientSecret);
    }
    confirmAuBecsDebitSetup(clientSecret, data) {
        return this.stripe.confirmAuBecsDebitSetup(clientSecret, data);
    }
    confirmCardSetup(clientSecret, data, options) {
        return this.stripe.confirmCardSetup(clientSecret, data, options);
    }
    confirmSepaDebitSetup(clientSecret, data) {
        return this.stripe.confirmSepaDebitSetup(clientSecret, data);
    }
    retrieveSetupIntent(clientSecret) {
        return this.stripe.retrieveSetupIntent(clientSecret);
    }
    paymentRequest(options) {
        return this.stripe.paymentRequest(options);
    }
    createToken(tokenType, data) {
        return this.stripe.createToken(tokenType, data);
    }
    createSource(a, b) {
        return this.stripe.createSource(a, b);
    }
    retrieveSource(source) {
        return this.stripe.retrieveSource(source);
    }
    /**
     * @deprecated
     */
    handleCardPayment(clientSecret, element, data) {
        return this.stripe.handleCardPayment(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    confirmPaymentIntent(clientSecret, element, data) {
        return this.stripe.confirmPaymentIntent(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    handleCardSetup(clientSecret, element, data) {
        return this.stripe.handleCardSetup(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    confirmSetupIntent(clientSecret, element, data) {
        return this.stripe.confirmSetupIntent(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    handleFpxPayment(clientSecret, element, data) {
        return this.stripe.handleFpxPayment(clientSecret, element, data);
    }
}
StripeService.decorators = [
    { type: Injectable }
];
StripeService.ctorParameters = () => [
    { type: String, decorators: [{ type: Inject, args: [NGX_STRIPE_VERSION,] }] },
    { type: String, decorators: [{ type: Inject, args: [STRIPE_PUBLISHABLE_KEY,] }] },
    { type: undefined, decorators: [{ type: Inject, args: [STRIPE_OPTIONS,] }] },
    { type: LazyStripeAPILoader },
    { type: WindowRef }
];
//# sourceMappingURL=data:application/json;base64,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