import { InjectionToken, Injectable, Inject, PLATFORM_ID, EventEmitter, Component, ViewChild, Input, Output, Directive, Optional, NgModule } from '@angular/core';
import { __awaiter } from 'tslib';
import { filter, first, map, switchMap } from 'rxjs/operators';
import { isPlatformBrowser, isPlatformServer } from '@angular/common';
import { BehaviorSubject, from } from 'rxjs';

const STRIPE_PUBLISHABLE_KEY = new InjectionToken('Stripe Publishable Key');
const STRIPE_OPTIONS = new InjectionToken('Stripe Options');
const NGX_STRIPE_VERSION = new InjectionToken('NGX_STRIPE_VERSION');

class WindowRef {
    constructor(platformId) {
        this.platformId = platformId;
    }
    getNativeWindow() {
        if (isPlatformBrowser(this.platformId)) {
            return window;
        }
        return {};
    }
}
WindowRef.decorators = [
    { type: Injectable }
];
WindowRef.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [PLATFORM_ID,] }] }
];

class DocumentRef {
    constructor(platformId) {
        this.platformId = platformId;
    }
    getNativeDocument() {
        if (isPlatformBrowser(this.platformId)) {
            return document;
        }
        return {};
    }
}
DocumentRef.decorators = [
    { type: Injectable }
];
DocumentRef.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [PLATFORM_ID,] }] }
];

class LazyStripeAPILoader {
    constructor(platformId, window, document) {
        this.platformId = platformId;
        this.window = window;
        this.document = document;
        this.status = new BehaviorSubject({
            error: false,
            loaded: false,
            loading: false
        });
    }
    asStream() {
        this.load();
        return this.status.asObservable();
    }
    isReady() {
        return this.status.getValue().loaded;
    }
    load() {
        if (isPlatformServer(this.platformId)) {
            return;
        }
        const status = this.status.getValue();
        if (this.window.getNativeWindow().hasOwnProperty('Stripe')) {
            this.status.next({
                error: false,
                loaded: true,
                loading: false
            });
        }
        else if (!status.loaded && !status.loading) {
            this.status.next(Object.assign(Object.assign({}, status), { loading: true }));
            const script = this.document.getNativeDocument().createElement('script');
            script.type = 'text/javascript';
            script.async = true;
            script.defer = true;
            script.src = 'https://js.stripe.com/v3/';
            script.onload = () => {
                this.status.next({
                    error: false,
                    loaded: true,
                    loading: false
                });
            };
            script.onerror = () => {
                this.status.next({
                    error: true,
                    loaded: false,
                    loading: false
                });
            };
            this.document.getNativeDocument().body.appendChild(script);
        }
    }
}
LazyStripeAPILoader.decorators = [
    { type: Injectable }
];
LazyStripeAPILoader.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [PLATFORM_ID,] }] },
    { type: WindowRef },
    { type: DocumentRef }
];

class StripeInstance {
    constructor(version, loader, window, key, options) {
        this.version = version;
        this.loader = loader;
        this.window = window;
        this.key = key;
        this.options = options;
        this.stripe$ = new BehaviorSubject(undefined);
        this.stripe = this.stripe$
            .asObservable()
            .pipe(filter((stripe) => Boolean(stripe)));
        this.loader
            .asStream()
            .pipe(filter((status) => status.loaded === true), first(), map(() => this.window.getNativeWindow().Stripe))
            .subscribe((stripeInstance) => {
            const stripe = this.options
                ? stripeInstance(this.key, this.options)
                : stripeInstance(this.key);
            stripe.registerAppInfo(this.getNgxStripeAppInfo(this.version));
            this.stripe$.next(stripe);
        });
    }
    getInstance() {
        return this.stripe$.getValue();
    }
    elements(options) {
        return this.stripe$.asObservable().pipe(filter((stripe) => Boolean(stripe)), map((stripe) => stripe.elements(options)), first());
    }
    redirectToCheckout(options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.redirectToCheckout(options))), first());
    }
    confirmAuBecsDebitPayment(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmAuBecsDebitPayment(clientSecret, data))), first());
    }
    confirmBancontactPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmBancontactPayment(clientSecret, data, options))), first());
    }
    confirmCardPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmCardPayment(clientSecret, data, options))), first());
    }
    confirmEpsPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmEpsPayment(clientSecret, data, options))), first());
    }
    confirmFpxPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmFpxPayment(clientSecret, data, options))), first());
    }
    confirmGiropayPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmGiropayPayment(clientSecret, data, options))), first());
    }
    confirmIdealPayment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmIdealPayment(clientSecret, data, options))), first());
    }
    confirmP24Payment(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmP24Payment(clientSecret, data, options))), first());
    }
    confirmSepaDebitPayment(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmSepaDebitPayment(clientSecret, data))), first());
    }
    handleCardAction(clientSecret) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleCardAction(clientSecret))), first());
    }
    createPaymentMethod(paymentMethodData) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.createPaymentMethod(paymentMethodData))), first());
    }
    retrievePaymentIntent(clientSecret) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.retrievePaymentIntent(clientSecret))), first());
    }
    confirmAuBecsDebitSetup(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmAuBecsDebitSetup(clientSecret, data))), first());
    }
    confirmCardSetup(clientSecret, data, options) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmCardSetup(clientSecret, data, options))), first());
    }
    confirmSepaDebitSetup(clientSecret, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmSepaDebitSetup(clientSecret, data))), first());
    }
    retrieveSetupIntent(clientSecret) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.retrieveSetupIntent(clientSecret))), first());
    }
    paymentRequest(options) {
        const stripe = this.getInstance();
        return stripe ? stripe.paymentRequest(options) : undefined;
    }
    createToken(tokenType, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.createToken(tokenType, data))), first());
    }
    createSource(a, b) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.createSource(a, b))), first());
    }
    retrieveSource(source) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.retrieveSource(source))), first());
    }
    /**
     * @deprecated
     */
    handleCardPayment(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleCardPayment(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    confirmPaymentIntent(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmPaymentIntent(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    handleCardSetup(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleCardSetup(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    confirmSetupIntent(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.confirmSetupIntent(clientSecret, element, data))), first());
    }
    /**
     * @deprecated
     */
    handleFpxPayment(clientSecret, element, data) {
        return this.stripe.pipe(switchMap((stripe) => from(stripe.handleFpxPayment(clientSecret, element, data))), first());
    }
    getNgxStripeAppInfo(version) {
        return {
            name: 'ngx-stripe',
            url: 'https://ngx-stripe.dev',
            partner_id: 'pp_partner_JR4l1rmvUoPP4V',
            version
        };
    }
}

class StripeService {
    constructor(version, key, options, loader, window) {
        this.version = version;
        this.key = key;
        this.options = options;
        this.loader = loader;
        this.window = window;
        if (key) {
            this.stripe = new StripeInstance(this.version, this.loader, this.window, key, options);
        }
    }
    getStripeReference() {
        return this.loader.asStream().pipe(filter((status) => status.loaded === true), map(() => this.window.getNativeWindow().Stripe));
    }
    getInstance() {
        return this.stripe.getInstance();
    }
    setKey(key, options) {
        return this.changeKey(key, options);
    }
    changeKey(key, options) {
        this.stripe = new StripeInstance(this.version, this.loader, this.window, key, options);
        return this.stripe;
    }
    elements(options) {
        return this.stripe.elements(options);
    }
    redirectToCheckout(options) {
        return this.stripe.redirectToCheckout(options);
    }
    confirmAuBecsDebitPayment(clientSecret, data) {
        return this.stripe.confirmAuBecsDebitPayment(clientSecret, data);
    }
    confirmBancontactPayment(clientSecret, data, options) {
        return this.stripe.confirmBancontactPayment(clientSecret, data, options);
    }
    confirmCardPayment(clientSecret, data, options) {
        return this.stripe.confirmCardPayment(clientSecret, data, options);
    }
    confirmEpsPayment(clientSecret, data, options) {
        return this.stripe.confirmEpsPayment(clientSecret, data, options);
    }
    confirmFpxPayment(clientSecret, data, options) {
        return this.stripe.confirmFpxPayment(clientSecret, data, options);
    }
    confirmGiropayPayment(clientSecret, data, options) {
        return this.stripe.confirmGiropayPayment(clientSecret, data, options);
    }
    confirmIdealPayment(clientSecret, data, options) {
        return this.stripe.confirmIdealPayment(clientSecret, data, options);
    }
    confirmP24Payment(clientSecret, data, options) {
        return this.stripe.confirmP24Payment(clientSecret, data, options);
    }
    confirmSepaDebitPayment(clientSecret, data) {
        return this.stripe.confirmSepaDebitPayment(clientSecret, data);
    }
    handleCardAction(clientSecret) {
        return this.stripe.handleCardAction(clientSecret);
    }
    createPaymentMethod(paymentMethodData) {
        return this.stripe.createPaymentMethod(paymentMethodData);
    }
    retrievePaymentIntent(clientSecret) {
        return this.stripe.retrievePaymentIntent(clientSecret);
    }
    confirmAuBecsDebitSetup(clientSecret, data) {
        return this.stripe.confirmAuBecsDebitSetup(clientSecret, data);
    }
    confirmCardSetup(clientSecret, data, options) {
        return this.stripe.confirmCardSetup(clientSecret, data, options);
    }
    confirmSepaDebitSetup(clientSecret, data) {
        return this.stripe.confirmSepaDebitSetup(clientSecret, data);
    }
    retrieveSetupIntent(clientSecret) {
        return this.stripe.retrieveSetupIntent(clientSecret);
    }
    paymentRequest(options) {
        return this.stripe.paymentRequest(options);
    }
    createToken(tokenType, data) {
        return this.stripe.createToken(tokenType, data);
    }
    createSource(a, b) {
        return this.stripe.createSource(a, b);
    }
    retrieveSource(source) {
        return this.stripe.retrieveSource(source);
    }
    /**
     * @deprecated
     */
    handleCardPayment(clientSecret, element, data) {
        return this.stripe.handleCardPayment(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    confirmPaymentIntent(clientSecret, element, data) {
        return this.stripe.confirmPaymentIntent(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    handleCardSetup(clientSecret, element, data) {
        return this.stripe.handleCardSetup(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    confirmSetupIntent(clientSecret, element, data) {
        return this.stripe.confirmSetupIntent(clientSecret, element, data);
    }
    /**
     * @deprecated
     */
    handleFpxPayment(clientSecret, element, data) {
        return this.stripe.handleFpxPayment(clientSecret, element, data);
    }
}
StripeService.decorators = [
    { type: Injectable }
];
StripeService.ctorParameters = () => [
    { type: String, decorators: [{ type: Inject, args: [NGX_STRIPE_VERSION,] }] },
    { type: String, decorators: [{ type: Inject, args: [STRIPE_PUBLISHABLE_KEY,] }] },
    { type: undefined, decorators: [{ type: Inject, args: [STRIPE_OPTIONS,] }] },
    { type: LazyStripeAPILoader },
    { type: WindowRef }
];

class StripeElementsService {
    constructor(stripeService) {
        this.stripeService = stripeService;
    }
    elements(stripe, options = {}) {
        if (stripe) {
            if (Object.keys(options).length > 0) {
                return stripe.elements(options);
            }
            return stripe.elements();
        }
        else {
            if (Object.keys(options).length > 0) {
                return this.stripeService.elements(options);
            }
            return this.stripeService.elements();
        }
    }
    paymentRequest(stripe, options) {
        return stripe
            ? stripe.paymentRequest(options)
            : this.stripeService.paymentRequest(options);
    }
    mergeOptions(options, containerClass) {
        if (!containerClass || (options && options.classes)) {
            return options || {};
        }
        if (!options || !options.classes) {
            return Object.assign(Object.assign({}, (options || {})), { classes: {
                    base: containerClass,
                    complete: `${containerClass}--complete`,
                    empty: `${containerClass}--empty`,
                    focus: `${containerClass}--focus`,
                    invalid: `${containerClass}--invalid`,
                    webkitAutoFill: `${containerClass}--webkit-autoFill`
                } });
        }
        return options || {};
    }
}
StripeElementsService.decorators = [
    { type: Injectable }
];
StripeElementsService.ctorParameters = () => [
    { type: StripeService }
];

class StripeCardComponent {
    constructor(stripeElementsService) {
        this.stripeElementsService = stripeElementsService;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
        this.state = 'notready';
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            this.state = 'starting';
            const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
            let updateElements = false;
            if (changes.elementsOptions || changes.stripe || !this.elements) {
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe, this.elementsOptions)
                    .toPromise();
                updateElements = true;
            }
            if (changes.options ||
                changes.containerClass ||
                !this.element ||
                updateElements) {
                if (this.element && !updateElements) {
                    this.update(options);
                }
                else if (this.elements && updateElements) {
                    this.createElement(options);
                }
            }
            this.state = 'ready';
        });
    }
    ngOnInit() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.state === 'notready') {
                this.state = 'starting';
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe)
                    .toPromise();
                this.createElement();
                this.state = 'ready';
            }
        });
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getCard() {
        return this.element;
    }
    createElement(options = {}) {
        this.element = this.elements.create('card', options);
        this.element.on('change', (ev) => this.change.emit(ev));
        this.element.on('blur', () => this.blur.emit());
        this.element.on('focus', () => this.focus.emit());
        this.element.on('ready', () => this.ready.emit());
        this.element.on('escape', () => this.escape.emit());
        this.element.mount(this.stripeElementRef.nativeElement);
        this.load.emit(this.element);
    }
}
StripeCardComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-card',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeCardComponent.ctorParameters = () => [
    { type: StripeElementsService }
];
StripeCardComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    elementsOptions: [{ type: Input }],
    stripe: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripeCardGroupDirective {
    constructor(stripeElementsService) {
        this.stripeElementsService = stripeElementsService;
        this.elements = new EventEmitter();
        this.state = 'notready';
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            this.state = 'starting';
            const elementsOptions = this.elementsOptions;
            const stripe = this.stripe;
            if (changes.elementsOptions || changes.stripe || !this._elements) {
                this._elements = yield this.stripeElementsService
                    .elements(stripe, elementsOptions)
                    .toPromise();
                this.elements.emit(this._elements);
            }
            this.state = 'ready';
        });
    }
    ngOnInit() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.state === 'notready') {
                this.state = 'starting';
                this._elements = yield this.stripeElementsService
                    .elements(this.stripe)
                    .toPromise();
                this.elements.emit(this._elements);
                this.state = 'ready';
            }
        });
    }
}
StripeCardGroupDirective.decorators = [
    { type: Directive, args: [{
                selector: 'ngx-stripe-card-group,[ngxStripeCardGroup]'
            },] }
];
StripeCardGroupDirective.ctorParameters = () => [
    { type: StripeElementsService }
];
StripeCardGroupDirective.propDecorators = {
    elementsOptions: [{ type: Input }],
    stripe: [{ type: Input }],
    elements: [{ type: Output }]
};

class StripeCardNumberComponent {
    constructor(stripeElementsService, cardGroup) {
        this.stripeElementsService = stripeElementsService;
        this.cardGroup = cardGroup;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            if (changes.options || changes.containerClass) {
                this.setupElement('options');
            }
        });
    }
    ngOnInit() {
        if (this.cardGroup) {
            this.cardGroupSubscription = this.cardGroup.elements.subscribe((elements) => {
                this.elements = elements;
                this.setupElement('elements');
            });
        }
        else {
            throw new Error('StripeCardNumberComponent must have StripeCardGroupDirective parent');
        }
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
        if (this.cardGroupSubscription) {
            this.cardGroupSubscription.unsubscribe();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getCardNumber() {
        return this.element;
    }
    setupElement(source) {
        const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
        if (this.element && source === 'options') {
            this.update(options);
        }
        else if (this.elements && source === 'elements') {
            this.element = this.elements.create('cardNumber', options);
            this.element.on('change', (ev) => this.change.emit(ev));
            this.element.on('blur', () => this.blur.emit());
            this.element.on('focus', () => this.focus.emit());
            this.element.on('ready', () => this.ready.emit());
            this.element.on('escape', () => this.escape.emit());
            this.element.mount(this.stripeElementRef.nativeElement);
            this.load.emit(this.element);
        }
    }
}
StripeCardNumberComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-card-number',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeCardNumberComponent.ctorParameters = () => [
    { type: StripeElementsService },
    { type: StripeCardGroupDirective, decorators: [{ type: Optional }] }
];
StripeCardNumberComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripeCardExpiryComponent {
    constructor(stripeElementsService, cardGroup) {
        this.stripeElementsService = stripeElementsService;
        this.cardGroup = cardGroup;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            if (changes.options || changes.containerClass) {
                this.setupElement('options');
            }
        });
    }
    ngOnInit() {
        if (this.cardGroup) {
            this.cardGroupSubscription = this.cardGroup.elements.subscribe((elements) => {
                this.elements = elements;
                this.setupElement('elements');
            });
        }
        else {
            throw new Error('StripeCardExpiryComponent must have StripeCardGroupDirective parent');
        }
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
        if (this.cardGroupSubscription) {
            this.cardGroupSubscription.unsubscribe();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getCardExpiry() {
        return this.element;
    }
    setupElement(source) {
        const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
        if (this.element && source === 'options') {
            this.update(options);
        }
        else if (this.elements && source === 'elements') {
            this.element = this.elements.create('cardExpiry', options);
            this.element.on('change', (ev) => this.change.emit(ev));
            this.element.on('blur', () => this.blur.emit());
            this.element.on('focus', () => this.focus.emit());
            this.element.on('ready', () => this.ready.emit());
            this.element.on('escape', () => this.escape.emit());
            this.element.mount(this.stripeElementRef.nativeElement);
            this.load.emit(this.element);
        }
    }
}
StripeCardExpiryComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-card-expiry',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeCardExpiryComponent.ctorParameters = () => [
    { type: StripeElementsService },
    { type: StripeCardGroupDirective, decorators: [{ type: Optional }] }
];
StripeCardExpiryComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripeCardCvcComponent {
    constructor(stripeElementsService, cardGroup) {
        this.stripeElementsService = stripeElementsService;
        this.cardGroup = cardGroup;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            if (changes.options || changes.containerClass) {
                this.setupElement('options');
            }
        });
    }
    ngOnInit() {
        if (this.cardGroup) {
            this.cardGroupSubscription = this.cardGroup.elements.subscribe((elements) => {
                this.elements = elements;
                this.setupElement('elements');
            });
        }
        else {
            throw new Error('StripeCardCvcComponent must have StripeCardGroupDirective parent');
        }
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
        if (this.cardGroupSubscription) {
            this.cardGroupSubscription.unsubscribe();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getCardCvc() {
        return this.element;
    }
    setupElement(source) {
        const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
        if (this.element && source === 'options') {
            this.update(options);
        }
        else if (this.elements && source === 'elements') {
            this.element = this.elements.create('cardCvc', options);
            this.element.on('change', (ev) => this.change.emit(ev));
            this.element.on('blur', () => this.blur.emit());
            this.element.on('focus', () => this.focus.emit());
            this.element.on('ready', () => this.ready.emit());
            this.element.on('escape', () => this.escape.emit());
            this.element.mount(this.stripeElementRef.nativeElement);
            this.load.emit(this.element);
        }
    }
}
StripeCardCvcComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-card-cvc',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeCardCvcComponent.ctorParameters = () => [
    { type: StripeElementsService },
    { type: StripeCardGroupDirective, decorators: [{ type: Optional }] }
];
StripeCardCvcComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripeFpxBankComponent {
    constructor(stripeElementsService) {
        this.stripeElementsService = stripeElementsService;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
        this.state = 'notready';
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            this.state = 'starting';
            const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
            let updateElements = false;
            if (changes.elementsOptions || changes.stripe || !this.elements) {
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe, this.elementsOptions)
                    .toPromise();
                updateElements = true;
            }
            if (changes.options ||
                changes.containerClass ||
                !this.element ||
                updateElements) {
                if (this.element && !updateElements) {
                    this.update(options);
                }
                else if (this.elements && updateElements) {
                    this.createElement(options);
                }
            }
            this.state = 'ready';
        });
    }
    ngOnInit() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.state === 'notready') {
                this.state = 'starting';
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe)
                    .toPromise();
                this.createElement();
                this.state = 'ready';
            }
        });
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getFpxBank() {
        return this.element;
    }
    createElement(options = { accountHolderType: 'individual' }) {
        this.element = this.elements.create('fpxBank', options);
        this.element.on('change', (ev) => this.change.emit(ev));
        this.element.on('blur', () => this.blur.emit());
        this.element.on('focus', () => this.focus.emit());
        this.element.on('ready', () => this.ready.emit());
        this.element.on('escape', () => this.escape.emit());
        this.element.mount(this.stripeElementRef.nativeElement);
        this.load.emit(this.element);
    }
}
StripeFpxBankComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-fpx-bank',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeFpxBankComponent.ctorParameters = () => [
    { type: StripeElementsService }
];
StripeFpxBankComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    elementsOptions: [{ type: Input }],
    stripe: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripeIdealBankComponent {
    constructor(stripeElementsService) {
        this.stripeElementsService = stripeElementsService;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
        this.state = 'notready';
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            this.state = 'starting';
            const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
            let updateElements = false;
            if (changes.elementsOptions || changes.stripe || !this.elements) {
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe, this.elementsOptions)
                    .toPromise();
                updateElements = true;
            }
            if (changes.options ||
                changes.containerClass ||
                !this.element ||
                updateElements) {
                if (this.element && !updateElements) {
                    this.update(options);
                }
                else if (this.elements && updateElements) {
                    this.createElement(options);
                }
            }
            this.state = 'ready';
        });
    }
    ngOnInit() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.state === 'notready') {
                this.state = 'starting';
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe)
                    .toPromise();
                this.createElement();
                this.state = 'ready';
            }
        });
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getIdealBank() {
        return this.element;
    }
    createElement(options = {}) {
        this.element = this.elements.create('idealBank', options);
        this.element.on('change', (ev) => this.change.emit(ev));
        this.element.on('blur', () => this.blur.emit());
        this.element.on('focus', () => this.focus.emit());
        this.element.on('ready', () => this.ready.emit());
        this.element.on('escape', () => this.escape.emit());
        this.element.mount(this.stripeElementRef.nativeElement);
        this.load.emit(this.element);
    }
}
StripeIdealBankComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-ideal-bank',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeIdealBankComponent.ctorParameters = () => [
    { type: StripeElementsService }
];
StripeIdealBankComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    elementsOptions: [{ type: Input }],
    stripe: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripeIbanComponent {
    constructor(stripeElementsService) {
        this.stripeElementsService = stripeElementsService;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
        this.state = 'notready';
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            this.state = 'starting';
            const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
            let updateElements = false;
            if (changes.elementsOptions || changes.stripe || !this.elements) {
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe, this.elementsOptions)
                    .toPromise();
                updateElements = true;
            }
            if (changes.options ||
                changes.containerClass ||
                !this.element ||
                updateElements) {
                if (this.element && !updateElements) {
                    this.update(options);
                }
                else if (this.elements && updateElements) {
                    this.createElement(options);
                }
            }
            this.state = 'ready';
        });
    }
    ngOnInit() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.state === 'notready') {
                this.state = 'starting';
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe)
                    .toPromise();
                this.createElement();
                this.state = 'ready';
            }
        });
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getIban() {
        return this.element;
    }
    createElement(options = {}) {
        this.element = this.elements.create('iban', options);
        this.element.on('change', (ev) => this.change.emit(ev));
        this.element.on('blur', () => this.blur.emit());
        this.element.on('focus', () => this.focus.emit());
        this.element.on('ready', () => this.ready.emit());
        this.element.on('escape', () => this.escape.emit());
        this.element.mount(this.stripeElementRef.nativeElement);
        this.load.emit(this.element);
    }
}
StripeIbanComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-iban',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeIbanComponent.ctorParameters = () => [
    { type: StripeElementsService }
];
StripeIbanComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    elementsOptions: [{ type: Input }],
    stripe: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripeAuBankAccountComponent {
    constructor(stripeElementsService) {
        this.stripeElementsService = stripeElementsService;
        this.load = new EventEmitter();
        this.blur = new EventEmitter();
        this.change = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.escape = new EventEmitter();
        this.state = 'notready';
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            this.state = 'starting';
            const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
            let updateElements = false;
            if (changes.elementsOptions || changes.stripe || !this.elements) {
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe, this.elementsOptions)
                    .toPromise();
                updateElements = true;
            }
            if (changes.options ||
                changes.containerClass ||
                !this.element ||
                updateElements) {
                if (this.element && !updateElements) {
                    this.update(options);
                }
                else if (this.elements && updateElements) {
                    this.createElement(options);
                }
            }
            this.state = 'ready';
        });
    }
    ngOnInit() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.state === 'notready') {
                this.state = 'starting';
                this.elements = yield this.stripeElementsService
                    .elements(this.stripe)
                    .toPromise();
                this.createElement();
                this.state = 'ready';
            }
        });
    }
    ngOnDestroy() {
        if (this.element) {
            this.element.destroy();
        }
    }
    update(options) {
        this.element.update(options);
    }
    /**
     * @deprecated
     */
    getAuBankAccount() {
        return this.element;
    }
    createElement(options = {}) {
        this.element = this.elements.create('auBankAccount', options);
        this.element.on('change', (ev) => this.change.emit(ev));
        this.element.on('blur', () => this.blur.emit());
        this.element.on('focus', () => this.focus.emit());
        this.element.on('ready', () => this.ready.emit());
        this.element.on('escape', () => this.escape.emit());
        this.element.mount(this.stripeElementRef.nativeElement);
        this.load.emit(this.element);
    }
}
StripeAuBankAccountComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-au-bank-account',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripeAuBankAccountComponent.ctorParameters = () => [
    { type: StripeElementsService }
];
StripeAuBankAccountComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    options: [{ type: Input }],
    elementsOptions: [{ type: Input }],
    stripe: [{ type: Input }],
    load: [{ type: Output }],
    blur: [{ type: Output }],
    change: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    escape: [{ type: Output }]
};

class StripePaymentRequestButtonComponent {
    constructor(stripeElementsService) {
        this.stripeElementsService = stripeElementsService;
        this.load = new EventEmitter();
        this.change = new EventEmitter();
        this.blur = new EventEmitter();
        this.focus = new EventEmitter();
        this.ready = new EventEmitter();
        this.token = new EventEmitter();
        this.paymentMethod = new EventEmitter();
        this.source = new EventEmitter();
        this.cancel = new EventEmitter();
        this.shippingaddresschange = new EventEmitter();
        this.shippingoptionchange = new EventEmitter();
        this.notavailable = new EventEmitter();
    }
    ngOnChanges(changes) {
        return __awaiter(this, void 0, void 0, function* () {
            const options = this.stripeElementsService.mergeOptions(this.options, this.containerClass);
            const elementsOptions = this.elementsOptions;
            const stripe = this.stripe;
            let updateElements = false;
            if (changes.elementsOptions || changes.stripe || !this.elements) {
                const elements = yield this.stripeElementsService
                    .elements(stripe, elementsOptions)
                    .toPromise();
                this.elements = elements;
                updateElements = true;
            }
            if (changes.paymentOptions && this.paymentRequest) {
                this.updateRequest(this.paymentOptions);
            }
            if (changes.options ||
                changes.containerClass ||
                !this.element ||
                updateElements) {
                if (this.element && !updateElements) {
                    this.update(options);
                }
                else if (this.elements && updateElements) {
                    this.paymentRequest = this.stripeElementsService.paymentRequest(stripe, this.paymentOptions);
                    this.paymentRequest.on('token', (ev) => this.token.emit(ev));
                    this.paymentRequest.on('paymentmethod', (ev) => this.paymentMethod.emit(ev));
                    this.paymentRequest.on('source', (ev) => this.source.emit(ev));
                    this.paymentRequest.on('cancel', () => this.cancel.emit());
                    this.paymentRequest.on('shippingaddresschange', (ev) => this.shippingaddresschange.emit(ev));
                    this.paymentRequest.on('shippingoptionchange', (ev) => this.shippingoptionchange.emit(ev));
                    this.element = this.elements.create('paymentRequestButton', Object.assign({ paymentRequest: this.paymentRequest }, options));
                    this.canMakePayment().subscribe((result) => {
                        if (result) {
                            this.element.on('click', (ev) => this.change.emit(ev));
                            this.element.on('blur', () => this.blur.emit());
                            this.element.on('focus', () => this.focus.emit());
                            this.element.on('ready', () => this.ready.emit());
                            this.element.mount(this.stripeElementRef.nativeElement);
                            this.load.emit({
                                paymentRequestButton: this.element,
                                paymentRequest: this.paymentRequest
                            });
                        }
                        else {
                            this.notavailable.emit();
                        }
                    });
                }
            }
        });
    }
    canMakePayment() {
        return from(this.paymentRequest.canMakePayment());
    }
    update(options) {
        this.element.update(options);
    }
    updateRequest(options) {
        const { currency, total, displayItems, shippingOptions } = options;
        this.paymentRequest.update({
            currency,
            total,
            displayItems,
            shippingOptions
        });
    }
    show() {
        this.paymentRequest.show();
    }
    /**
     * @deprecated
     */
    getButton() {
        return this.element;
    }
}
StripePaymentRequestButtonComponent.decorators = [
    { type: Component, args: [{
                selector: 'ngx-stripe-payment-request-button',
                template: `<div class="field" #stripeElementRef></div>`
            },] }
];
StripePaymentRequestButtonComponent.ctorParameters = () => [
    { type: StripeElementsService }
];
StripePaymentRequestButtonComponent.propDecorators = {
    stripeElementRef: [{ type: ViewChild, args: ['stripeElementRef',] }],
    containerClass: [{ type: Input }],
    paymentOptions: [{ type: Input }],
    options: [{ type: Input }],
    elementsOptions: [{ type: Input }],
    stripe: [{ type: Input }],
    load: [{ type: Output }],
    change: [{ type: Output }],
    blur: [{ type: Output }],
    focus: [{ type: Output }],
    ready: [{ type: Output }],
    token: [{ type: Output }],
    paymentMethod: [{ type: Output }],
    source: [{ type: Output }],
    cancel: [{ type: Output }],
    shippingaddresschange: [{ type: Output }],
    shippingoptionchange: [{ type: Output }],
    notavailable: [{ type: Output }]
};

class StripeFactoryService {
    constructor(version, baseKey, baseOptions, loader, window) {
        this.version = version;
        this.baseKey = baseKey;
        this.baseOptions = baseOptions;
        this.loader = loader;
        this.window = window;
    }
    create(key, options) {
        if (!key && !this.baseKey) {
            return null;
        }
        return new StripeInstance(this.version, this.loader, this.window, key || this.baseKey, options || this.baseOptions);
    }
}
StripeFactoryService.decorators = [
    { type: Injectable }
];
StripeFactoryService.ctorParameters = () => [
    { type: String, decorators: [{ type: Inject, args: [NGX_STRIPE_VERSION,] }] },
    { type: String, decorators: [{ type: Inject, args: [STRIPE_PUBLISHABLE_KEY,] }] },
    { type: undefined, decorators: [{ type: Inject, args: [STRIPE_OPTIONS,] }] },
    { type: LazyStripeAPILoader },
    { type: WindowRef }
];

const components = [
    StripeCardComponent,
    StripeCardNumberComponent,
    StripeCardExpiryComponent,
    StripeCardCvcComponent,
    StripeFpxBankComponent,
    StripeIdealBankComponent,
    StripeIbanComponent,
    StripeAuBankAccountComponent,
    StripePaymentRequestButtonComponent
];
const directives = [StripeCardGroupDirective];
const currentVersion = '12.0.1';
class NgxStripeModule {
    static forRoot(publishableKey, options) {
        return {
            ngModule: NgxStripeModule,
            providers: [
                LazyStripeAPILoader,
                StripeService,
                StripeFactoryService,
                StripeElementsService,
                WindowRef,
                DocumentRef,
                {
                    provide: STRIPE_PUBLISHABLE_KEY,
                    useValue: publishableKey
                },
                {
                    provide: STRIPE_OPTIONS,
                    useValue: options
                },
                {
                    provide: NGX_STRIPE_VERSION,
                    useValue: currentVersion
                }
            ]
        };
    }
    static forChild(publishableKey, options) {
        return {
            ngModule: NgxStripeModule,
            providers: [
                LazyStripeAPILoader,
                StripeService,
                StripeFactoryService,
                StripeElementsService,
                WindowRef,
                DocumentRef,
                {
                    provide: STRIPE_PUBLISHABLE_KEY,
                    useValue: publishableKey
                },
                {
                    provide: STRIPE_OPTIONS,
                    useValue: options
                },
                {
                    provide: NGX_STRIPE_VERSION,
                    useValue: currentVersion
                }
            ]
        };
    }
}
NgxStripeModule.decorators = [
    { type: NgModule, args: [{
                declarations: [...components, ...directives],
                exports: [...components, ...directives]
            },] }
];

/*
 * Public API Surface of ngx-stripe
 */

/**
 * Generated bundle index. Do not edit.
 */

export { DocumentRef, LazyStripeAPILoader, NgxStripeModule, STRIPE_OPTIONS, STRIPE_PUBLISHABLE_KEY, StripeAuBankAccountComponent, StripeCardComponent, StripeCardCvcComponent, StripeCardExpiryComponent, StripeCardGroupDirective, StripeCardNumberComponent, StripeElementsService, StripeFactoryService, StripeFpxBankComponent, StripeIbanComponent, StripeIdealBankComponent, StripeInstance, StripePaymentRequestButtonComponent, StripeService, WindowRef, NGX_STRIPE_VERSION as ɵa };
//# sourceMappingURL=ngx-stripe.js.map
