import winston from 'winston';
import fs from 'fs';

const colors = {
  error: 'red',
  warn: 'yellow',
  info: 'green',
  http: 'magenta',
  debug: 'white',
};
winston.addColors(colors);

// let logFolder: string = '';
function currentDate(): string {
  return new Date().toISOString().split('T')[0];
}
// let currentDate = new Date().toISOString().split('T')[0];

function getUnauthorizedLogTransport() {
  const logFolder = `logs/${currentDate()}`;

  if (!fs.existsSync(logFolder)) {
    fs.mkdirSync(logFolder, { recursive: true });
  }

  return new winston.transports.File({
    filename: `${logFolder}/unauthorized-access.log`,
    level: 'warn',
  });
}

// Function to replace spaces in keys with underscores
function sanitizeKeys(obj: any): any {
  if (typeof obj !== 'object' || obj === null) return obj;

  if (Array.isArray(obj)) {
    return obj.map(sanitizeKeys);
  }

  const sanitizedObj: any = {};
  for (const key in obj) {
    const sanitizedKey = key.includes(' ') ? key.replace(/ /g, '_') : key;
    sanitizedObj[sanitizedKey] = sanitizeKeys(obj[key]);
  }

  return sanitizedObj;
}

// Masking function to show only the first and last letter of the password
function maskSensitiveData(data: any): any {
  const sanitizedData = sanitizeKeys(JSON.parse(JSON.stringify(data)));

  const maskFields = (obj: any) => {
    for (const key in obj) {
      if (typeof obj[key] === 'object' && obj[key] !== null) {
        maskFields(obj[key]);
      } else if (key.toLowerCase() === 'password' && typeof obj[key] === 'string') {
        const password = obj[key];
        if (password.length > 2) {
          // Reveal only the first and last characters of the password
          obj[key] = password[0] + '*'.repeat(password.length - 2) + password[password.length - 1];
        } else {
          obj[key] = '*'.repeat(password.length); // Fully mask if password length is 2 or less
        }
      }
    }
  };

  maskFields(sanitizedData);
  return sanitizedData;
}

const format = winston.format.combine(
  winston.format.json(),
  winston.format.timestamp({ format: 'MM-DD-YYYY HH:mm:ss:ms' }),
  winston.format.printf(({ timestamp, message }) => {
    // Sanitize keys in the log message before processing
    const sanitizedMessage = sanitizeKeys(JSON.parse(message));
    const messageString = JSON.stringify(sanitizedMessage);

    const idaccountRegex = /"idaccount":(\d+)/;
    const match = messageString.match(idaccountRegex);
    const idaccountValue = match ? match[1] : '';
    const filteredMessage = messageString.replace(idaccountRegex, '').replace(/,,/g, ',');

    if (idaccountValue && !isNaN(parseInt(idaccountValue))) {
      checkNewLogFile(idaccountValue, filteredMessage);
    }

    return filteredMessage;
  })
);

function checkNewLogFile(idaccountValue: string, message: string) {
  const logFolder = `logs/${currentDate()}`;

  try {
    if (!fs.existsSync(logFolder)) {
      fs.mkdirSync(logFolder, { recursive: true });
    }

    const idaccountFilename = `${logFolder}/${100000000 + parseInt(idaccountValue)}.log`;
    if (!fs.existsSync(idaccountFilename)) {
      fs.writeFileSync(idaccountFilename, '');
    }

    const logData = `${message}\n`;
    fs.appendFileSync(idaccountFilename, logData);
  } catch (error) {
    console.error('Error occurred while creating or writing to log file:', error);
  }
}

const Logger = winston.createLogger({
  format,
  transports: [
    new winston.transports.Console({ level: 'error' }),
    getUnauthorizedLogTransport(),
  ],
});

export { Logger, maskSensitiveData };
