const express = require('express');
const bodyParser = require('body-parser');
const mysql = require('mysql');
const cron = require('node-cron');
const { DB_CONFIG } = require('../../trunk/dbconfig/dbconnect');

const app = express();
app.use(bodyParser.json());

// Create database pool
const pool = mysql.createPool({
    connectionLimit: DB_CONFIG.connectionLimit,
    host: DB_CONFIG.host,
    user: DB_CONFIG.user,
    password: DB_CONFIG.password,
    database: DB_CONFIG.database,
    port: DB_CONFIG.port
});

// Handle pool errors
pool.on('error', (err) => {
    console.error('Pool error:', err.message);
    if (err.code === 'PROTOCOL_CONNECTION_LOST' || 
        err.code === 'ECONNRESET' || 
        err.code === 'PROTOCOL_ENQUEUE_AFTER_FATAL_ERROR') {
        console.log('Attempting to reconnect to database...');
        setTimeout(() => {
            pool.end();
            pool = mysql.createPool(DB_CONFIG);
        }, 5000);
    }
});

// Custom error handler
process.on('uncaughtException', (err) => {
    console.error('Uncaught Exception:', err);
});

// Handle unhandled promise rejections
process.on('unhandledRejection', (reason, promise) => {
    console.error('Unhandled Rejection:', reason);
});

// Graceful shutdown
process.on('SIGTERM', () => {
    console.log('SIGTERM received. Closing pool...');
    pool.end((err) => {
        if (err) {
            console.error('Error closing pool:', err.message);
        }
        console.log('Pool closed successfully');
        process.exit(err ? 1 : 0);
    });
});

process.on('SIGINT', () => {
    console.log('SIGINT received. Closing pool...');
    pool.end((err) => {
        if (err) {
            console.error('Error closing pool:', err.message);
        }
        console.log('Pool closed successfully');
        process.exit(err ? 1 : 0);
    });
});

function migrateSCMDRLive() {
    console.log(`\n=====================================================\n\tStarting Shortcode MDR Live Migration ${new Date().toLocaleString()}\n=====================================================\n`);

    pool.getConnection((err, connection) => {
        if (err) {
            console.error('Error getting connection from pool:', err.message);
            return;
        }

        // Get data from mdrlive_shortcode
        const selectQuery = `SELECT idmdrlive, idaccount, iduser, idapi, batch_id, number_type, to_number, from_number, 
            messsage_body, status, message_accepted_by_signalmash, message_delivered_to_operator,
            delivery_receipt_received_by_singnalmash, delivery_receipt_delivered, npa, state, 
            MO_rates, MT_rates, MT_surcharge, MO_surcharge, connection_key, sms_outbound_key, 
            direction, type, msg_count, mobility, ocn, lata, category, mdr_status, idoperator, 
            operator, dlr_err, dlr_stat, did_provider, dca, campaignId, cost, is_billable 
            FROM mdrlive_shortcode WHERE DATE_SUB(NOW(), INTERVAL 72 HOUR) > created_at LIMIT 5000`;

        connection.query(selectQuery, (error, results) => {
            if (error) {
                console.error('Error fetching from mdrlive_shortcode:', error.message);
                connection.release();
                return;
            }

            if (results.length === 0) {
                console.log("No records found in mdrlive_shortcode");
                connection.release();
                return;
            }

            console.log(`Found ${results.length} records to process`);

            let processedCount = 0;
            const totalRecords = results.length;

            // Process each record
            results.forEach((record) => {
                // Insert into mdr
                let status=99, dlr_err='000', dlr_stat='EXPIRED', mcc=0, mnc=0, src_dcaid=0, dsc_dcaid=0;
                
                const mdrQuery = `INSERT INTO mdr (idaccount, iduser, idapi, batch_id, number_type, to_number, 
                    from_number, messsage_body, status, message_accepted_by_signalmash, 
                    message_delivered_to_operator, delivery_receipt_received_by_singnalmash, 
                    delivery_receipt_delivered, npa, state, MO_rates, MT_rates, 
                    MT_surcharge, MO_surcharge, connection_key, sms_outbound_key, 
                    direction, type, msg_count, mobility, ocn, lata, category, 
                    mdr_status, idoperator, operator, dlr_err, dlr_stat, did_provider, 
                    dca, campaignId, cost, is_billable, mcc, mnc, src_dcaid, dsc_dcaid, idmdrlive ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;

                const mdrValues = [
                    record.idaccount, record.iduser, record.idapi, record.batch_id,
                    record.number_type, record.to_number, record.from_number,
                    record.messsage_body, status,
                    record.message_accepted_by_signalmash, record.message_delivered_to_operator,
                    record.delivery_receipt_received_by_singnalmash, record.delivery_receipt_delivered,
                    record.npa, record.state, record.MO_rates, record.MT_rates,
                    record.MT_surcharge, record.MO_surcharge, record.connection_key,
                    record.sms_outbound_key, record.direction, record.type, record.msg_count,
                    record.mobility, record.ocn, record.lata, record.category,
                    record.mdr_status, record.idoperator, record.operator, dlr_err,
                    dlr_stat, record.did_provider, record.dca,
                    record.campaignId, record.cost, record.is_billable, mcc, mnc, src_dcaid, dsc_dcaid, record.idmdrlive
                ];

                connection.query(mdrQuery, mdrValues, (errMdr, respMdr) => {
                    if (errMdr) {
                        console.error(`Failed to insert into mdr for idmdrlive ${record.idmdrlive}:`, errMdr.message);
                        processedCount++;
                        checkCompletion();
                        return;
                    }

                    console.log(`Inserted into mdr for idmdrlive: ${record.idmdrlive}`);

                    // Delete from mdrlive_shortcode
                    const deleteQuery = 'DELETE FROM mdrlive_shortcode WHERE idmdrlive = ?';
                    connection.query(deleteQuery, [record.idmdrlive], (errDelete, respDelete) => {
                        if (errDelete) {
                            console.error(`Failed to delete from mdrlive_shortcode for idmdrlive ${record.idmdrlive}:`, errDelete.message);
                        } else {
                            console.log(`Deleted from mdrlive_shortcode for idmdrlive: ${record.idmdrlive}`);
                        }
                        processedCount++;
                        checkCompletion();
                    });
                });
            });

            function checkCompletion() {
                if (processedCount === totalRecords) {
                    console.log(`Completed processing ${totalRecords} records`);
                    connection.release();
                }
            }
        });
    });
}
// Schedule migration to run daily at 1 AM
cron.schedule('0 1 * * *', () => {
    migrateSCMDRLive();
});

// Start server
const PORT = 4702;
app.listen(PORT, () => {
    console.log(`Shortcode Migration server started at Port: ${PORT}`);
}); 