"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.maskSensitiveData = exports.Logger = void 0;
const winston_1 = __importDefault(require("winston"));
const fs_1 = __importDefault(require("fs"));
const colors = {
    error: 'red',
    warn: 'yellow',
    info: 'green',
    http: 'magenta',
    debug: 'white',
};
winston_1.default.addColors(colors);
let logFolder = '';
const currentDate = new Date().toISOString().split('T')[0];
function getUnauthorizedLogTransport() {
    logFolder = `logs/${currentDate}`;
    if (!fs_1.default.existsSync(logFolder)) {
        fs_1.default.mkdirSync(logFolder, { recursive: true });
    }
    return new winston_1.default.transports.File({
        filename: `${logFolder}/unauthorized-access.log`,
        level: 'warn',
    });
}
// Function to replace spaces in keys with underscores
function sanitizeKeys(obj) {
    if (typeof obj !== 'object' || obj === null)
        return obj;
    if (Array.isArray(obj)) {
        return obj.map(sanitizeKeys);
    }
    const sanitizedObj = {};
    for (const key in obj) {
        const sanitizedKey = key.includes(' ') ? key.replace(/ /g, '_') : key;
        sanitizedObj[sanitizedKey] = sanitizeKeys(obj[key]);
    }
    return sanitizedObj;
}
// Masking function to show only the first and last letter of the password
function maskSensitiveData(data) {
    const sanitizedData = sanitizeKeys(JSON.parse(JSON.stringify(data)));
    const maskFields = (obj) => {
        for (const key in obj) {
            if (typeof obj[key] === 'object' && obj[key] !== null) {
                maskFields(obj[key]);
            }
            else if (key.toLowerCase() === 'password' && typeof obj[key] === 'string') {
                const password = obj[key];
                if (password.length > 2) {
                    // Reveal only the first and last characters of the password
                    obj[key] = password[0] + '*'.repeat(password.length - 2) + password[password.length - 1];
                }
                else {
                    obj[key] = '*'.repeat(password.length); // Fully mask if password length is 2 or less
                }
            }
        }
    };
    maskFields(sanitizedData);
    return sanitizedData;
}
exports.maskSensitiveData = maskSensitiveData;
const format = winston_1.default.format.combine(winston_1.default.format.json(), winston_1.default.format.timestamp({ format: 'MM-DD-YYYY HH:mm:ss:ms' }), winston_1.default.format.printf(({ timestamp, message }) => {
    // Sanitize keys in the log message before processing
    const sanitizedMessage = sanitizeKeys(typeof message === 'string' ? JSON.parse(message) : message);
    const messageString = JSON.stringify(sanitizedMessage);
    const idaccountRegex = /"idaccount":(\d+)/;
    const match = messageString.match(idaccountRegex);
    const idaccountValue = match ? match[1] : '';
    const filteredMessage = messageString.replace(idaccountRegex, '').replace(/,,/g, ',');
    if (idaccountValue && !isNaN(parseInt(idaccountValue))) {
        checkNewLogFile(idaccountValue, filteredMessage);
    }
    return filteredMessage;
}));
function checkNewLogFile(idaccountValue, message) {
    logFolder = `logs/${currentDate}`;
    try {
        if (!fs_1.default.existsSync(logFolder)) {
            fs_1.default.mkdirSync(logFolder, { recursive: true });
        }
        const idaccountFilename = `${logFolder}/${100000000 + parseInt(idaccountValue)}.log`;
        if (!fs_1.default.existsSync(idaccountFilename)) {
            fs_1.default.writeFileSync(idaccountFilename, '');
        }
        const logData = `${message}\n`;
        fs_1.default.appendFileSync(idaccountFilename, logData);
    }
    catch (error) {
        console.error('Error occurred while creating or writing to log file:', error);
    }
}
const Logger = winston_1.default.createLogger({
    format,
    transports: [
        new winston_1.default.transports.Console({ level: 'error' }),
        getUnauthorizedLogTransport(),
    ],
});
exports.Logger = Logger;
