import { GaxiosOptions, GaxiosPromise, GaxiosResponse } from 'gaxios';
import { BodyResponseCallback } from '../transporters';
import { Credentials } from './credentials';
import { AuthClient } from './authclient';
import { GetAccessTokenResponse, Headers, RefreshOptions } from './oauth2client';
/**
 * Offset to take into account network delays and server clock skews.
 */
export declare const EXPIRATION_TIME_OFFSET: number;
/**
 * Internal interface for tracking the access token expiration time.
 */
interface CredentialsWithResponse extends Credentials {
    res?: GaxiosResponse | null;
}
/**
 * Defines an upper bound of permissions available for a GCP credential.
 */
interface CredentialAccessBoundary {
    accessBoundary: {
        accessBoundaryRules: AccessBoundaryRule[];
    };
}
/** Defines an upper bound of permissions on a particular resource. */
interface AccessBoundaryRule {
    availablePermissions: string[];
    availableResource: string;
    availabilityCondition?: AvailabilityCondition;
}
/**
 * An optional condition that can be used as part of a
 * CredentialAccessBoundary to further restrict permissions.
 */
interface AvailabilityCondition {
    expression: string;
    title?: string;
    description?: string;
}
export declare class DownscopedClient extends AuthClient {
    private client;
    private cab;
    /**
     * OAuth scopes for the GCP access token to use. When not provided,
     * the default https://www.googleapis.com/auth/cloud-platform is
     * used.
     */
    private cachedDownscopedAccessToken;
    private readonly stsCredential;
    readonly authClient: AuthClient;
    readonly credentialAccessBoundary: CredentialAccessBoundary;
    readonly eagerRefreshThresholdMillis: number;
    readonly forceRefreshOnFailure: boolean;
    constructor(client: AuthClient, cab: CredentialAccessBoundary, additionalOptions?: RefreshOptions);
    /**
     * Provides a mechanism to inject Downscoped access tokens directly.
     * When the provided credential expires, a new credential, using the
     * external account options are retrieved.
     * Notice DownscopedClient is the broker class mainly used for generate
     * downscoped access tokens, it is unlikely we call this function in real
     * use case.
     * We implement to make this a helper function for testing all cases in getAccessToken().
     * @param credentials The Credentials object to set on the current client.
     */
    setCredentials(credentials: Credentials): void;
    getAccessToken(): Promise<GetAccessTokenResponse>;
    /**
     * The main authentication interface. It takes an optional url which when
     * present is the endpoint> being accessed, and returns a Promise which
     * resolves with authorization header fields.
     *
     * The result has the form:
     * { Authorization: 'Bearer <access_token_value>' }
     */
    getRequestHeaders(): Promise<Headers>;
    /**
     * Provides a request implementation with OAuth 2.0 flow. In cases of
     * HTTP 401 and 403 responses, it automatically asks for a new access token
     * and replays the unsuccessful request.
     * @param opts Request options.
     * @param callback callback.
     * @return A promise that resolves with the HTTP response when no callback
     * is provided.
     */
    request<T>(opts: GaxiosOptions): GaxiosPromise<T>;
    request<T>(opts: GaxiosOptions, callback: BodyResponseCallback<T>): void;
    /**
     * Forces token refresh, even if unexpired tokens are currently cached.
     * GCP access tokens are retrieved from authclient object/source credential.
     * Thenm GCP access tokens are exchanged for downscoped access tokens via the
     * token exchange endpoint.
     * @return A promise that resolves with the fresh downscoped access token.
     */
    protected refreshAccessTokenAsync(): Promise<CredentialsWithResponse>;
    /**
     * Returns whether the provided credentials are expired or not.
     * If there is no expiry time, assumes the token is not expired or expiring.
     * @param downscopedAccessToken The credentials to check for expiration.
     * @return Whether the credentials are expired or not.
     */
    private isExpired;
}
export {};
