import axios, { AxiosRequestConfig, AxiosResponse } from 'axios';
import FormData from 'form-data';
 
export interface RequestOptions {
  method?: string;                        // HTTP method: GET, POST, PUT, DELETE, PATCH
  url?: string;                           // API URL (made optional to match request package)
  headers?: Record<string, any>;          // Optional headers (made more flexible)
  body?: any;                             // Optional body for POST, PUT, PATCH
  params?: Record<string, any>;           // Optional query params for GET/DELETE
  log?: boolean;                          // Optional logging flag (default false)
  auth?: { user: string; pass: string };  // Optional auth (for request compatibility)
  form?: any;                             // Optional form data (for request compatibility)
  qs?: Record<string, any>;               // Optional query string (for request compatibility)
  formData?: Record<
    string,
    | string
    | number
    | boolean
    | Buffer
    | NodeJS.ReadableStream
    | {
        value: NodeJS.ReadableStream | Buffer | string;
        options?: {
          filename?: string;
          contentType?: string | null;
        };
      }
  >;
}

export interface RequestResponse {
  statusCode: number;
  headers: Record<string, string>;
  body: string;
}

// Type definitions for different callback patterns
type Callback2Params = (error: any, response?: RequestResponse) => void;
type Callback3Params = (error: any, response?: RequestResponse, body?: any) => void;
type FlexibleCallback = Callback2Params | Callback3Params;

/**
 * Detects the number of parameters expected by a callback function
 * @param callback The callback function to analyze
 * @returns The number of parameters the callback expects
 */
function detectCallbackParams(callback: Function): number {
  const callbackStr = callback.toString();
  const paramMatch = callbackStr.match(/\([^)]*\)/);
  
  if (!paramMatch) return 3; // Default to 3 params for safety
  
  const params = paramMatch[0].slice(1, -1).split(',').map(p => p.trim()).filter(p => p.length > 0);
  
  // If callback has exactly 2 parameters, return 2
  if (params.length === 2) return 2;
  
  // Default to 3 parameters for compatibility
  return 3;
}

/**
 * Enhanced request function that handles both 2-param and 3-param callback patterns
 * @param options Request configuration options
 * @param callback Callback function (supports both 2 and 3 parameter patterns)
 */
export function request(
  options: RequestOptions,
  callback: FlexibleCallback
): void {
  if (options.log) {
    console.log(`[AxiosRequest] ${options.method || 'GET'} ${options.url}`);
  }

  const method = (options.method || 'GET').toLowerCase();
  const config: AxiosRequestConfig = {
    method: method as any,
    url: options.url || '',
    headers: options.headers || {},
    params: options.params || options.qs,
    validateStatus: () => true,
    timeout: 30000 // 30 second timeout
  };

  // Handle auth if provided
  if (options.auth) {
    config.auth = {
      username: options.auth.user,
      password: options.auth.pass
    };
  }

  // Handle formData (file uploads)

  if (options.formData) {
    const formData = new FormData();
    for (const key in options.formData) {
      const value = options.formData[key];
      if (value && typeof value === 'object' && 'value' in value) {
        // Handle file stream
        formData.append(key, value.value, value.options || {});
      } else {
        // Normal field
        formData.append(key, value as any);
      }
    }
    config.data = formData;
    config.headers = { ...config.headers, ...formData.getHeaders() };
  }
  // Handle x-www-form-urlencoded
  else if (options.form) {
    config.data = options.form;
    config.headers = { ...config.headers, 'Content-Type': 'application/x-www-form-urlencoded' };
  } else if (options.body && ['post', 'put', 'patch', 'delete'].includes(method)) {
    if (typeof options.body === 'string') {
      try {
        config.data = JSON.parse(options.body);
        config.headers = { ...config.headers, 'Content-Type': 'application/json' };
      } catch (error) {
        // If JSON parsing fails, send the raw string as data
        config.data = options.body;
        config.headers = { ...config.headers, 'Content-Type': 'text/plain' };
      }
    } else {
      config.data = options.body;
      config.headers = { ...config.headers, 'Content-Type': 'application/json' };
    }
  }

  // Detect callback parameter count
  const callbackParamCount = detectCallbackParams(callback);

  axios(config)
    .then((response: AxiosResponse) => {
      const res: RequestResponse = {
        statusCode: response.status,
        headers: response.headers as Record<string, string>,
        body: typeof response.data === 'string' ? response.data : JSON.stringify(response.data)
      };

      if (options.log) {
        console.log(`[AxiosRequest] SUCCESS: ${method.toUpperCase()} ${options.url} -> ${res.statusCode}`);
      }

      // Call callback with appropriate number of parameters
      if (callbackParamCount === 2) {
        (callback as Callback2Params)(null, res);
      } else {
        (callback as Callback3Params)(null, res, res.body);
      }
    })
    .catch((error: any) => {
      const errorMessage = error.response?.data?.message || error.message || 'Unknown error';
      
      if (options.log) {
        console.error(`[AxiosRequest] ERROR: ${method.toUpperCase()} ${options.url}`, errorMessage);
      }

      // Call callback with appropriate number of parameters
      if (callbackParamCount === 2) {
        (callback as Callback2Params)(error, undefined);
      } else {
        (callback as Callback3Params)(error, undefined, undefined);
      }
  });
}
