const express = require("express");
const { connection, key } = require("./dbcon/db");
const bodyparser = require("body-parser");
const HashMap = require("hashmap");
const app = express();
const allSUrChargeRates = new HashMap();
app.use(bodyparser.json());

// Memory usage helper function
function getMemoryUsage() {
    const memUsage = process.memoryUsage();
    return {
        rss: Math.round(memUsage.rss / 1024 / 1024), // MB
        heapTotal: Math.round(memUsage.heapTotal / 1024 / 1024), // MB
        heapUsed: Math.round(memUsage.heapUsed / 1024 / 1024), // MB
        external: Math.round(memUsage.external / 1024 / 1024), // MB
        arrayBuffers: Math.round(memUsage.arrayBuffers / 1024 / 1024) // MB
    };
}

// LOADING time UTC 9:00 AM
// To Load LRN DIDs in HashMap
function loadLergeInfo() {
    console.log("=== Starting LERG Info Load ===");
    const startTime = Date.now();
    const startMemory = getMemoryUsage();
    
    console.log("Memory before load:", startMemory);
    
    getAllSurchargeRates(function (err, resSurcharge) {
        const endTime = Date.now();
        const endMemory = getMemoryUsage();
        const executionTime = endTime - startTime;
        
        if (err === 1 && resSurcharge.length > 0) {
            for (let i = 0; i < resSurcharge.length; i++) {
                allSUrChargeRates.set(resSurcharge[i].npa + '_' + resSurcharge[i].nxx + '_' + resSurcharge[i].direction + '_' + resSurcharge[i].subtype + '_' + resSurcharge[i].category + '_' + resSurcharge[i].type, resSurcharge[i]);
            }
            
            const memoryDiff = {
                rss: endMemory.rss - startMemory.rss,
                heapTotal: endMemory.heapTotal - startMemory.heapTotal,
                heapUsed: endMemory.heapUsed - startMemory.heapUsed,
                external: endMemory.external - startMemory.external,
                arrayBuffers: endMemory.arrayBuffers - startMemory.arrayBuffers
            };
            
            console.log("=== LERG Info Load Complete ===");
            console.log("Execution time:", executionTime, "ms");
            console.log("Records loaded:", resSurcharge.length);
            console.log("HashMap size:", allSUrChargeRates.size);
            console.log("Memory after load:", endMemory);
            console.log("Memory difference:", memoryDiff);
            console.log("Surcharge Rates HashMap Set:", allSUrChargeRates);
        }
        else {
            console.log("=== LERG Info Load Failed ===");
            console.log("Execution time:", executionTime, "ms");
            console.log("Memory after load:", endMemory);
            console.info("No Record Available in surcharge_rates");
        }
    });
}

// To Get LRN DIDs
function getAllSurchargeRates(callback) {
    const queryStartTime = Date.now();
    let qry = `SELECT sr.direction, sr.subtype, sr.category, sr.type, npa, nxx, BLOCK, lata, ocn, lata, state, co_name_spec_c, li.category, IFNULL(o.id,0) idoperator, IFNULL(sr.rates,0) rates,IFNULL(o.name,'') name FROM (
	    SELECT * FROM lerg_info
    ) li
    LEFT JOIN lerg_operator lo ON lo.operator=li.co_name_spec_c AND lo.category=li.category
    LEFT JOIN operator_mapping om ON om.idlerg_operator=lo.idlerg_operator
    LEFT JOIN operator o ON om.idoperator=o.id
    LEFT JOIN (
        SELECT * FROM surcharge_rates WHERE direction='outbound' AND category='none'
    ) sr ON sr.idoperator=o.id;`;
    
    console.log("LERG Info Query Load:", qry);
    console.log("Query start time:", new Date().toISOString());
    
    connection.execute(qry, (err, rows) => {      
        const queryEndTime = Date.now();
        const queryExecutionTime = queryEndTime - queryStartTime;
        
        console.log("Query execution time:", queryExecutionTime, "ms");
        console.log("Query end time:", new Date().toISOString());
        console.log("LERG Info Query Result:", rows);
        console.log("Records returned:", rows ? rows.length : 0);
        
        if(err==1){
            callback(1, rows);
        }
        else{
            callback(0, {status: 500, message: "Internal Server Error", error: err.message });
        }    
    });
}

// Route: To Get LRN DID Detail By DID with custom number type
app.get('/getLrnSurcharge/:did/:direction/:subtype/:category/:numbertype', (req, res) => {
    let token = req.headers.authorization || req.headers.token;
    if (token !== key) {
        return res.status(401).send({
            message: "Invalid Session you are trying to access",
            error: "Invalid session is called to access!"
        });
    }

    const { did, direction, subtype, category, numbertype } = req.params;
    console.log("Requested LERG Info:", { did, direction, subtype, category, numbertype });
    
    // Check cache first
    const cacheKey = `${did.substring(0,3)}_${did.substring(3,6)}_${direction}_${subtype}_${category}_${numbertype}`;
    const cachedData = allSUrChargeRates.get(cacheKey);
    
    if (cachedData) {
        console.log("Cache hit for key:", cacheKey);
        return res.status(200).send({ status: 200, message: "LERG Info Found in Cache!", data: cachedData });
    }
    else{
        console.log("Cache miss for key:", cacheKey);
        const queryStartTime = Date.now();
        
        let qry = `SELECT npa,block, lata, ocn, lata, state, co_name_spec_c, li.category, IFNULL(o.id,0) idoperator, IFNULL(sr.rates,0) rates,IFNULL(o.name,'') name FROM (
            SELECT * FROM lerg_info
            WHERE npa=SUBSTRING('${did}', (LENGTH('${did}')-9), 3) AND nxx=SUBSTRING('${did}', (LENGTH('${did}')-6), 3)
            LIMIT 1
        ) li
        LEFT JOIN lerg_operator lo ON lo.operator=li.co_name_spec_c AND lo.category=li.category
        LEFT JOIN operator_mapping om ON om.idlerg_operator=lo.idlerg_operator
        LEFT JOIN operator o ON om.idoperator=o.id
        LEFT JOIN (
            SELECT * FROM surcharge_rates
            WHERE direction='${direction}' AND subtype='${subtype}' AND category='${category}' AND type='${numbertype}'
        ) sr ON sr.idoperator=o.id`;
        
        console.log("Query Get LRN Info:", qry);
        connection.execute(qry, (err, rows) => {
            const queryEndTime = Date.now();
            const queryExecutionTime = queryEndTime - queryStartTime;
            
            console.log("Individual query execution time:", queryExecutionTime, "ms");
            
            if(err == 1) {
                if(rows.length > 0) {
                    allSUrChargeRates.set(cacheKey, rows[0]);
                    console.log("Added to cache with key:", cacheKey);
                    res.status(200).send({ status: 200, message: "LERG Info Found!", data: rows[0]});
                } else {
                    res.status(404).send({ status: 404, message: "LERG Info Not Found!", data: {}});
                }
            }
            else{
                res.status(500).send({ status: 500, message: "Internal Server Error"});
            }
        });
    }
});

// Memory status endpoint
app.get('/memory-status', (req, res) => {
    const memoryUsage = getMemoryUsage();
    const hashMapSize = allSUrChargeRates.size;
    
    res.status(200).send({
        status: 200,
        message: "Memory Status",
        data: {
            memory: memoryUsage,
            hashMapSize: hashMapSize,
            hashMapKeys: Array.from(allSUrChargeRates.keys()).slice(0, 10), // Show first 10 keys
            timestamp: new Date().toISOString()
        }
    });
});

// Schedule LERG info loading at UTC 9 AM daily
function scheduleLergInfoLoad() {
    const now = new Date();
    const utc9am = new Date(Date.UTC(now.getUTCFullYear(), now.getUTCMonth(), now.getUTCDate(), 9, 0, 0));
    
    // If it's already past 9 AM today, schedule for tomorrow
    if (now.getTime() > utc9am.getTime()) {
        utc9am.setUTCDate(utc9am.getUTCDate() + 1);
    }
    
    const timeUntilNext = utc9am.getTime() - now.getTime();
    
    console.log(`Next LERG info load scheduled for: ${utc9am.toISOString()}`);
    console.log(`Time until next load: ${Math.floor(timeUntilNext / 1000 / 60)} minutes`);
    
    setTimeout(() => {
        loadLergeInfo();
        // Schedule the next load for tomorrow
        scheduleLergInfoLoad();
    }, timeUntilNext);
}

// Start the scheduling
scheduleLergInfoLoad();

app.listen(3002, () => {
    loadLergeInfo();
    console.log("LERG Info Server is connected at port 3002");
    console.log("LERG info will be loaded daily at UTC 9:00 AM");
    console.log("Memory monitoring enabled - use /memory-status endpoint to check current status");
}); 