const express = require("express");
const { connection } = require("./dbcon/db");
const bodyparser = require("body-parser");
const HashMap = require("hashmap");

const app = express();
app.use(bodyparser.json());

// Campaign Service for HashMap Management
class CampaignService {
    constructor() {
        this.campaignData = new HashMap();
    }

    fetchCampaignData() {
        this.campaignData.clear(); 
        const qry = `SELECT c.idbrand, c.campaign_key, cm.tpm, cm.mno, cm.mmsTpm as mmstpm, cm.brandDailycap FROM campaign c INNER JOIN campaign_mno cm ON c.campaign_key = cm.campaign_key WHERE c.deleted = 0 AND c.provisioning_status = '3' AND c.campaign_key IS NOT NULL AND c.campaign_key != ''`;
        console.log("Fetching Campaign Data Query:", qry);
        connection.execute(qry, (err, rows) => {
            if (err != 1) {
                console.error("Error fetching campaign data:", err);
                return;
            }
            if (rows.length > 0) {
                // Count unique campaigns for progress tracking
                const uniqueCampaigns = new Set();
                rows.forEach(row => uniqueCampaigns.add(row.campaign_key));
                
                console.log(`[DATA LOAD] Processing ${rows.length} campaign-MNO combinations from ${uniqueCampaigns.size} campaigns`);
                
                rows.forEach((row, index) => {
                    let key = row.mno.toLowerCase() == 't-mobile' ? row.idbrand + '_' + row.mno : row.campaign_key + '_' + row.mno;

                    let value = {
                        campaignKey: row.campaign_key,
                        brandId: row.idbrand,
                        tpm: (row.tpm && parseInt(row.tpm) > 0) ? parseInt(row.tpm) : 0,
                        mno: row.mno,
                        mmstpm: (row.mmstpm && parseInt(row.mmstpm) > 0) ? parseInt(row.mmstpm) : 0,
                        brandDailycap: (row.brandDailycap && parseInt(row.brandDailycap) > 0) ? parseInt(row.brandDailycap) : 0
                    };

                    console.log(`Processing ${index + 1} of ${rows.length}: ${row.campaign_key}_${row.mno}`);
                    this.campaignData.set(key, value);
                    console.log(`HashMap updated for campaign_key ${key}`, value);
                    console.log(`HashMap Size: ${this.campaignData.size}`);
                });
                
                console.log(`[DATA LOAD] Completed! Loaded ${this.campaignData.size} entries into HashMap`);
            }
        });
    }
}

const campaignService = new CampaignService();

// AUTO-REFRESH CAMPAIGN DATA EVERY 5 MINUTES
setInterval(() => {
    console.log(`[AUTO-REFRESH] Refreshing campaign data at ${new Date().toISOString()}`);
    campaignService.fetchCampaignData();
}, 5 * 60 * 1000); // 5 minutes

// Initial data load
console.log(`[INITIAL LOAD] Loading campaign data at startup...`);
campaignService.fetchCampaignData();

// CUSTOM RATE LIMITER - Optimized for thousands of unique keys
class CustomRateLimiter {
  constructor() {
    this.requests = new Map(); // Store request counts per key
    this.windows = new Map();  // Store window start times per key
    this.cleanupInterval = 5 * 60 * 1000; // Cleanup every 5 minutes
    this.maxInactiveTime = 24 * 60 * 60 * 1000; // Remove keys inactive for 24 hours (for daily limits)
    
    // Start periodic cleanup
    this.startCleanup();
  }

  startCleanup() {
    setInterval(() => {
      this.cleanupInactiveKeys();
    }, this.cleanupInterval);
  }

  cleanupInactiveKeys() {
    const now = Date.now();
    let cleanedCount = 0;
    
    console.log(`[CLEANUP] Starting cleanup of inactive keys...`);
    
    // Clean up old requests
    for (let [key, requests] of this.requests) {
      const windowStart = this.windows.get(key);
      if (!windowStart || (now - windowStart) > this.maxInactiveTime) {
        this.requests.delete(key);
        this.windows.delete(key);
        cleanedCount++;
      } else {
        // Clean old request timestamps within active windows
        const validRequests = requests.filter(timestamp => {
          if (key.includes('brandDailycap')) {
            // For daily limits, keep requests from current calendar day
            const requestDate = new Date(timestamp);
            const today = new Date();
            const windowStart = this.windows.get(key);
            
            // Only keep requests from the current calendar day window
            return timestamp >= windowStart && requestDate.toDateString() === today.toDateString();
          } else {
            // For minute limits, keep requests from last hour
            return timestamp > now - (60 * 60 * 1000);
          }
        });
        
        if (validRequests.length === 0) {
          this.requests.delete(key);
          this.windows.delete(key);
          cleanedCount++;
        } else if (validRequests.length !== requests.length) {
          this.requests.set(key, validRequests);
        }
      }
    }
    
    if (cleanedCount > 0) {
      console.log(`[CLEANUP] Cleaned up ${cleanedCount} inactive keys. Active keys: ${this.requests.size}`);
    }
  }


  // Check if request should be allowed
  checkLimit(key, maxRequests, windowMs, limitType = 'tpm') {
    const now = Date.now();
    const windowKey = `${key}_${limitType}`; // Unique key per limit type
    
    // Get existing window start, or create new window if none exists
    let windowStart = this.windows.get(windowKey);
    
    if (limitType === 'brandDailycap') {
      // For daily limits, use calendar day (12 AM to 12 AM)
      const today = new Date();
      today.setHours(0, 0, 0, 0); // Set to midnight
      const todayMidnight = today.getTime();
      
      if (!windowStart || windowStart < todayMidnight) {
        windowStart = todayMidnight;
        this.windows.set(windowKey, windowStart);
        this.requests.set(windowKey, []); // Clear requests for new day
        console.log(`[RATE LIMITER] New calendar day started, resetting daily window for ${key} (${limitType})`);
      }
    } else {
      // For TPM/MMSTPM, use rolling windows
      if (!windowStart) {
        windowStart = now;
        this.windows.set(windowKey, windowStart);
        console.log(`[RATE LIMITER] Creating new ${limitType} window for ${key} starting at ${windowStart}`);
      }
    }
    
    // Check if we need to reset the window (fixed window approach)
    let currentRequests;
    const timeElapsed = now - windowStart;
    const windowType = limitType === 'brandDailycap' ? '24 hours' : '1 minute';
    
    if (timeElapsed >= windowMs) {
      console.log(`[RATE LIMITER] ${windowType} window expired, resetting for ${key} (${limitType})`);
      currentRequests = [];
      this.requests.set(windowKey, currentRequests);
      this.windows.set(windowKey, now);
    } else {
      currentRequests = this.requests.get(windowKey) || [];
    }
    
    // Calculate reset time based on limit type
    let resetTime;
    if (limitType === 'brandDailycap') {
      // For daily limits, reset time is next midnight
      const tomorrow = new Date();
      tomorrow.setDate(tomorrow.getDate() + 1);
      tomorrow.setHours(0, 0, 0, 0);
      resetTime = tomorrow.getTime();
    } else {
      // For TPM/MMSTPM, use rolling window
      resetTime = windowStart + windowMs;
    }

    // Check if limit exceeded BEFORE adding current request
    if (currentRequests.length >= maxRequests) {
      console.log(`[CUSTOM RATE LIMITER] Rate limit exceeded for ${key}: ${currentRequests.length}/${maxRequests}`);
      return {
        allowed: false,
        remaining: 0,
        resetTime: resetTime,
        currentCount: currentRequests.length,
        limitType: limitType
      };
    }

    // Add current request
    currentRequests.push(now);
    this.requests.set(windowKey, currentRequests);

    const windowTypeDisplay = limitType === 'brandDailycap' ? 'per day (calendar)' : 'per minute';
    console.log(`[CUSTOM RATE LIMITER] Request allowed for ${key}: ${currentRequests.length}/${maxRequests} ${windowTypeDisplay}`);
    return {
      allowed: true,
      remaining: maxRequests - currentRequests.length,
      resetTime: resetTime,
      currentCount: currentRequests.length,
      limitType: limitType
    };
  }
}

const customRateLimiter = new CustomRateLimiter();

// MAIN RATE LIMIT CHECK ENDPOINT
app.post("/campaign/limitcheck/:key", (req, res) => {
  // Step 1: Extract and validate request data
  const key = req.params.key; // Expected: "CAMPAIGN_MNO_TYPE" (e.g., "CZBNXRQ_T-Mobile_sms")
  console.log("Key:: ", key);
  
  // Validate required fields
  if (!key) {
    return res.status(400).json({ 
      status: 400, 
      message: "Invalid request",
      details: "Required field: key (in URL)"
    });
  }
  
  // Step 2: Parse the URL key to extract campaign information
  const keyParts = key.split('_');
  if (keyParts.length < 3) {
    return res.status(400).json({ 
      status: 400, 
      message: "Invalid key format",
      details: "Key should be in format: CAMPAIGN_MNO_TYPE or BRANDID_MNO_TYPE"
    });
  }
  
  const mno = keyParts[keyParts.length - 2];        // Mobile Network Operator
  const type = keyParts[keyParts.length - 1];       // Message type (sms, mms)
  const campaignOrBrand = keyParts.slice(0, -2).join('_'); // Campaign or Brand ID
  
  console.log(`[Rate Limit Check] Key: ${key}, Type: ${type}, MNO: ${mno}, Campaign/Brand: ${campaignOrBrand}`);

  // Step 3: Look up campaign data from HashMap
  const hashMapKey = `${campaignOrBrand}_${mno}`;
  const campaignData = campaignService.campaignData.get(hashMapKey);
  console.log("Campaign Data:: ", campaignData);
  
  if (!campaignData) {
    return res.status(400).json({ 
      status: 400, 
      message: `SMS cannot be sent for this number`,
      details: `No data found in HashMap for key: ${hashMapKey}`
    });
  }
  
  // Step 4: Determine rate limit based on message type
  let limitValue = Number.MAX_SAFE_INTEGER;
  let limitType = '';
  let windowMs = 60 * 1000; // Default 1 minute
  
  if (type == "sms") {
    if (mno.toLowerCase() == 't-mobile') {
      // T-Mobile uses brandDailycap for SMS
      if (campaignData.brandDailycap == 0) {
        limitValue = Number.MAX_SAFE_INTEGER;
        limitType = 'brandDailycap_unlimited';
      } else {
        limitValue = campaignData.brandDailycap;
        limitType = 'brandDailycap';
        windowMs = 24 * 60 * 60 * 1000; // 24 hours
      }
    } else {
      // Other carriers use TPM for SMS
      if (campaignData.tpm == 0) {
        limitValue = Number.MAX_SAFE_INTEGER;
        limitType = 'tpm_unlimited';
      } else {
        limitValue = campaignData.tpm;
        limitType = 'tpm';
        windowMs = 60 * 1000; // 1 minute
      }
    }
  } else if (type == "mms") {
    if (campaignData.mmstpm == 0) {
      limitValue = Number.MAX_SAFE_INTEGER;
      limitType = 'mmstpm_unlimited';
    } else {
      limitValue = campaignData.mmstpm;
      limitType = 'mmstpm';
      windowMs = 60 * 1000; // 1 minute
    }
  } else {
    return res.status(400).json({ 
      status: 400, 
      message: "Invalid message type",
      details: "Type must be 'sms' or 'mms'"
    });
  }

  // Step 5: Apply custom rate limiting
  const rateLimitResult = customRateLimiter.checkLimit(key, limitValue, windowMs, limitType);
  
  if (!rateLimitResult.allowed) {
    console.log(`[RATE LIMIT BLOCKED] ${type.toUpperCase()} message blocked for ${key} (${rateLimitResult.limitType})`);
    return res.status(429).json({
      status: 429,
      /* message: `${mno} ${type.toUpperCase()} rate limit exceeded (${limitValue} messages per ${rateLimitResult.limitType == 'brandDailycap' ? 'day' : 'minute'})`, */
      message: `${type.toUpperCase()} rate limit exceeded (${limitValue} messages per ${rateLimitResult.limitType == 'brandDailycap' ? 'day' : 'minute'})`,
      data: {
        key: key,
        limit: limitValue,
        limitType: rateLimitResult.limitType,
        used: rateLimitResult.currentCount,
        remaining: rateLimitResult.remaining,
        resetTime: new Date(rateLimitResult.resetTime).toISOString(),
        windowMs: windowMs,
        windowType: rateLimitResult.limitType == 'brandDailycap' ? '24 hours' : '1 minute'
      }
    });
  }

  // Step 6: Set headers and log success
  res.set({
    'X-RateLimit-Type': type,
    'X-RateLimit-MNO': mno,
    'X-RateLimit-Key': key,
    'X-RateLimit-LimitType': limitType,
    'X-RateLimit-Limit': limitValue,
    'X-RateLimit-Remaining': rateLimitResult.remaining,
    'X-RateLimit-Reset': Math.ceil(rateLimitResult.resetTime / 1000)
  });

  console.log(`[SMS/MMS SENT] ${type.toUpperCase()} message sent successfully`);
  console.log(`[RATE LIMIT STATUS] Key: ${key}`);
  console.log(`[RATE LIMIT STATUS] MNO: ${mno}, Type: ${type.toUpperCase()}`);
  console.log(`[RATE LIMIT STATUS] Limit Type: ${limitType}`);
  console.log(`[RATE LIMIT STATUS] Total Limit: ${limitValue == Number.MAX_SAFE_INTEGER ? 'Unlimited' : limitValue}`);
  console.log(`[RATE LIMIT STATUS] Used: ${rateLimitResult.currentCount}, Remaining: ${rateLimitResult.remaining}`);
  console.log(`[RATE LIMIT STATUS] Window: ${limitType == 'brandDailycap' ? '24 hours' : '1 minute'}`);
  console.log(`[RATE LIMIT STATUS] Reset Time: ${new Date(rateLimitResult.resetTime).toISOString()}`);
  console.log(`[RATE LIMIT STATUS] ==========================================`);

  // Step 7: Return success response
  res.status(200).json({ 
    status: 200, 
    message: "Rate limit check passed",
    data: {
      key: key,
      hashMapKey: hashMapKey,
      campaignOrBrand: campaignOrBrand,
      limit: limitValue,
      limitType: limitType,
      type: type,
      mno: mno,
      currentUsage: {
        used: rateLimitResult.currentCount,
        remaining: rateLimitResult.remaining,
        limit: limitValue,
        windowMs: windowMs,
        resetTime: new Date(rateLimitResult.resetTime).toISOString()
      },
      campaignData: {
        campaignKey: campaignData.campaignKey,
        brandId: campaignData.brandId,
        tpm: campaignData.tpm,
        mmstpm: campaignData.mmstpm,
        brandDailycap: campaignData.brandDailycap
      }
    }
  });
});

app.listen(4500, () => {
    console.log("Server running on port 4500");
    console.log("SMS Rate Limiting System initialized");
    console.log("");
    console.log("Available endpoint:");
    console.log("  POST /campaign/limitcheck/:key - Rate limiting endpoint");
    console.log("  POST /campaign/refresh - Manual campaign data refresh");
    console.log("");
    console.log("Auto-refresh: Campaign data refreshes every 5 minutes");
    console.log("");
    console.log("Example usage:");
    console.log("  POST /campaign/limitcheck/CMA45JB_AT&T_sms");
    console.log("  POST /campaign/limitcheck/CMA45JB_Verizon Wireless_sms");
    console.log("");
    console.log("Payload structure:");
    console.log("  No payload required - all data extracted from URL key");
    console.log("");
    console.log("Rate Limiting:");
    console.log("  - Uses custom rate limiter with HashMap data");
    console.log("  - Dynamic limits: TPM for SMS, MMSTPM for MMS");
    console.log("  - Unlimited sending: When limits are 0 or invalid");
    console.log("  - Data source: campaignService.campaignData HashMap");
});
