const parquet = require('parquetjs-lite');
const fs = require('fs');
const path = require('path');
const { connection } = require("../db");
const basepath = "/var/www/html/panel/reports/summary/";


async function getCDRMaxid(connection) {
  try {
    const query = `SELECT MAX(idcdr)idcdr FROM cdr`

    const [rows] = await connection.execute(query);

    if(rows.length>0)
      return rows[0].idcdr;
    else
      return 0;
  } catch (error) {
    console.error('Error CDR Max ID:', error);
    throw error;
  }
}

async function getCDRFromid(connection) {
  try {
    const query = `SELECT idrecord FROM report_summary_index WHERE rtype=1 AND status=1 LIMIT 1`;

    const [rows] = await connection.execute(query);
    if(rows.length>0){
      await updateIndexStatus(connection, 0);
      return rows[0].idrecord;
    }      
    else
      return 0;
  } catch (error) {
    console.error('Error CDR From ID:', error);
    throw error;
  }
}

async function updateIndex(connection, maxid) {
  try {
    const query = `UPDATE report_summary_index SET idrecord='${maxid}', status=1 WHERE rtype=1`;

    const [rows] = await connection.execute(query);

    return rows;
  } catch (error) {
    console.error('Error update CDR Last Index:', error);
    throw error;
  }
}

async function updateIndexStatus(connection, status) {
  try {
    const query = `UPDATE report_summary_index SET status=${status} WHERE rtype=1`;

    const [rows] = await connection.execute(query);

    return rows;
  } catch (error) {
    console.error('Error update mdr Last Index:', error);
    throw error;
  }
}

/* // Function to generate the CDR summary report from the database without filters */
async function generateCdrSummaryReport(connection, to, from) {
  try {
    /* // SQL query to fetch summary data, grouped by account_id, date, san, and direction */
    const query = `SELECT IFNULL(DATE(start_stamp),'0') AS date, IFNULL(accountcode,'0') idaccount, IFNULL(iduser,'0')iduser,  direction,  COUNT(*) AS total_calls,SUM(CASE WHEN hangup_cause='NORMAL_CLEARING' THEN 1 ELSE 0 END) AS success, SUM(CASE WHEN hangup_cause!='NORMAL_CLEARING' THEN 1 ELSE 0 END) AS failed, IFNULL(SUM(cost),0.00) AS total_cost, IFNULL(SUM(duration),0) AS duration FROM cdr WHERE idcdr>=${from} AND idcdr<${to} GROUP BY direction, date, accountcode, iduser ORDER BY date DESC`
    console.log(query);
    const [rows] = await connection.execute(query);

    return rows;
  } catch (error) {
    console.error('Error generating CDR summary report:', error);
    throw error;
  }
}

/* // Function to write the summary data to a Parquet file in a dynamic directory */
async function writeSummaryToParquet(summary) {
  try {
    const schema = new parquet.ParquetSchema({
      date: { type: 'UTF8' },
      idaccount: { type: 'UTF8' },
      iduser: { type: 'UTF8' },
      direction: { type: 'UTF8' },
      total_calls: { type: 'INT64' },
      success: { type: 'INT64' },
      failed: { type: 'INT64' },
      duration: { type: 'DOUBLE' },
      total_cost: { type: 'DOUBLE' },
    });

    const { date, idaccount } = summary;
    summary.idaccount = summary.idaccount.toString();
    summary.iduser = summary.iduser.toString();
    const san = 100000000 + parseInt(accountcode, 10) || 0;
    const outputDir = path.join(basepath, 'cdr', `${san}`);
    if (!fs.existsSync(outputDir)) {
      fs.mkdirSync(outputDir, { recursive: true });
    }

    const filename = `${date.replace(/-/g, '')}.parquet`; 
    const outputFilePath = path.join(dirPath, filename);

    const writer = await parquet.ParquetWriter.openFile(schema, outputFilePath);
    console.log("Data :: ", summary);
    await writer.appendRow(summary);
    await writer.close();

    console.log(`Summary report for account ${idaccount} on ${date} written to ${outputFilePath}`);
  } catch (error) {
    console.error('Error writing summary to Parquet file:', error);
    throw error;
  }
}

/* // Main function to execute the script */
async function main() {
  try {

    const maxid = await getCDRMaxid(connection);
    const fromid = await getCDRFromid(connection);

    if(fromid>0){
      /* // Generate the CDR summary report without filters */
      const summaries = await generateCdrSummaryReport(connection, maxid, fromid);

      const uindex = await updateIndex(connection, maxid);
      /* // Iterate through each summary record and write it to the corresponding Parquet file */
      for (const summary of summaries) {
        console.log("Data22 :: ", summary);
        await writeSummaryToParquet(summary);
      }
    }
    else{
      await updateIndexStatus(connection, 1);
      console.log("CDR summary are under process");
    }
  } catch (error) {
    console.error('Error in main execution:', error);
  } finally {
    if (connection) {
      /* // Close the database connection */
      await connection.end();
    }
  }
}

/* // Run the script */
main();